/**
 * @file batchadmin.cc
 *
 * BatchAdmin - Class for the administration ob batch jobs.
 *
 * Copyright: 2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: batchadmin.cc,v 1.5.4.1 2005/05/27 16:28:08 thomas Exp $
 * Project  CAPI for BSD
 * Target   capitest - Test tool for the functionality of CAPI for BSD
 * @date    27.08.2003
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

// System includes
#include <fstream>
#include <sstream>
#include <stdexcept>

// Import includes

#define __BATCHADMIN__

// Local includes
#ifndef __BATCHADMIN_H
#  include "batchadmin.h"
#endif





// === Private definitions ===============================================





// === Definitions for public declarations ===============================





// === Prototypes for private functions ==================================





// === Implementation of class members ===================================





// --- Class for the administration of batch jobs ------------------------

/**
 * Constructor for a batch file.
 */

CBatchAdmin::CBatchAdmin
   (const std::string &strBatchFileName)
{
   ReadBatchFile (strBatchFileName);
   if (m_aJobs.size () == 0)
   {
      throw std::runtime_error ("Batch file does not contain any valid job line");
   }
   
} // CBatchAdmin::CBatchAdmin





/**
 * Constructor to create a batch of identical jobs.
 */

CBatchAdmin::CBatchAdmin
   (size_t                               nNumCalls,
    unsigned                             uController,
    CCapiStateMachine::ConnectionType_t  connType,
    const std::string                   &strSrcFileName,
    const std::string                   &strDstPhoneNumber,
    const std::string                   &strSrcPhoneNumber)
{
   // create a job entry to be duplicated nNumCalls times in the jobs array
   JobData_t newJob;
   newJob.uController       = uController;
   newJob.connType          = connType;
   newJob.strSrcFileName    = strSrcFileName;
   newJob.strDstPhoneNumber = strDstPhoneNumber;
   newJob.strSrcPhoneNumber = strSrcPhoneNumber;
   
   m_aJobs.clear ();
   m_aJobs.resize (nNumCalls, newJob);
   
} // CBatchAdmin::CBatchAdmin





/**
 * Get the number of calls in the batch.
 *
 * The jobs maintained in this batch object have ids from null up to but
 * excluding the result of this function.
 */

size_t CBatchAdmin::GetNumCalls (void)
{
   return (m_aJobs.size ());
} // CBatchAdmin::GetNumCalls





/**
 * Get the data for the job with a specific id.
 */

void CBatchAdmin::GetJobData
   (unsigned                             uJobID,
    unsigned                            &uController,
    CCapiStateMachine::ConnectionType_t &connType,
    std::string                         &strSrcFileName,
    std::string                         &strDstPhoneNumber,
    std::string                         &strSrcPhoneNumber)
{
   JobData_t *pJob;
   
   pJob = &(m_aJobs [uJobID % m_aJobs.size ()]);
   uController       = pJob->uController;
   connType          = pJob->connType;
   strSrcFileName    = pJob->strSrcFileName;
   strDstPhoneNumber = pJob->strDstPhoneNumber;
   strSrcPhoneNumber = pJob->strSrcPhoneNumber;
   
} // CBatchAdmin::GetJobData





/**
 * Read a batch file and construct the job array from it.
 */

void CBatchAdmin::ReadBatchFile
   (const std::string &strBatchFileName)
{
   std::ifstream is (strBatchFileName.c_str ());
   std::string   strLine;
   
   while (! is.eof () && ! is.fail ())
   {
      strLine.erase ();
      (void) std::getline (is, strLine, '\n');
      if (strLine.length () > 0)
      {
         EvalBatchLine (strLine);
      }
   }
   
} // CBatchAdmin::ReadBatchFile





/**
 * Evaluate a line of a batch file and create a job entry from it.
 */

void CBatchAdmin::EvalBatchLine
   (const std::string &strLine)
{
   // check for empty or comment line
   if (strLine.length () == 0)
   {
      return;
   }
   if (strLine [0] == '#' || strLine [0] == ';')
   {
      return;
   }
   
   // check for a line containing only spaces and / or tabs
   if (strLine.find_first_not_of (" \t\r") == std::string::npos)
   {
      return;
   }
   
   JobData_t          newJob;
   std::istringstream iss (strLine);
   std::string        strConnType;
   
   iss >> newJob.uController >> strConnType >> newJob.strSrcFileName
       >> newJob.strDstPhoneNumber;
   if (iss.fail ())
   {
      throw std::runtime_error ("Error in batch file line");
   }
   iss >> newJob.strSrcPhoneNumber;
   
   if (strConnType == "voice" || strConnType == "v")
   {
      newJob.connType = CCapiStateMachine::CT_VOICE;
   }
   else if (strConnType == "hdlc" || strConnType == "h")
   {
      newJob.connType = CCapiStateMachine::CT_DATA_HDLC;
   }
   else if (strConnType == "x75" || strConnType == "x")
   {
      newJob.connType = CCapiStateMachine::CT_DATA_X75;
   }
   else if (strConnType == "fax" || strConnType == "f")
   {
      newJob.connType = CCapiStateMachine::CT_FAXG3;
   }
   else
   {
      throw std::runtime_error ("Invalid connection type in batch file");
   }
   
   m_aJobs.resize (m_aJobs.size () + 1, newJob);
   
} // CBatchAdmin::EvalBatchLine





// === Implementation of public functions ================================





// === Implementation of private functions ===============================
