/**
 * @file batchadmin.h
 *
 * BatchAdmin - Class for the administration ob batch jobs.
 *
 * Copyright: 2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: batchadmin.h,v 1.3.4.1 2005/05/27 16:28:08 thomas Exp $
 * Project  CAPI for BSD
 * Target   capitest - Test tool for the functionality of CAPI for BSD
 * @date    27.08.2003
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __BATCHADMIN_H
#define __BATCHADMIN_H

#include <string>
#include <vector>

#include "capism.h"





// === Public declarations ===============================================





// === Class declarations ================================================





/**
 * Class for the administration of batch jobs.
 *
 * A batch of jobs can be created in two ways. The first is through a batch file
 * the name of which is specified in the constructor call. The batch file is
 * read line by line. Every line that does not start with a '#' or a ';' will be
 * assumed to have five fields separated by spaces and / or tabs. They are
 * interpreted as the controller number, connection type ("voice", "hdlc",
 * "x75", "fax"), source file name, destination phone number and source phone
 * number. The last field may be left out. If any error occurs evaluating the
 * batch file, the constructor will throw a runtime exception.
 *
 * The other way to create a batch of jobs is to create a list with every
 * element containing the same data. I.e. the batch will consist of the
 * specified number of entries and each entry is made up of the values also
 * specified in the constructor call.
 *
 * Every job is associated with a job id, starting at null. The code part using
 * this class is expected to maintain an array or the like of calls with a
 * length equal to the number of entries stored in this object. The caller
 * associates the call array index with the id maintainted for the batch job. So
 * we need not take care of jobs running or not in objects of this class.
 */
class CBatchAdmin
{
   public:
      /**
       * Constructor for a batch file.
       */
      CBatchAdmin
         (const std::string &strBatchFileName);

      /**
       * Constructor to create a batch of identical jobs.
       */
      CBatchAdmin
         (size_t                               nNumCalls,
          unsigned                             uController,
          CCapiStateMachine::ConnectionType_t  connType,
          const std::string                   &strSrcFileName,
          const std::string                   &strDstPhoneNumber,
          const std::string                   &strSrcPhoneNumber);
      
      /**
       * Get the number of calls in the batch.
       *
       * The jobs maintained in this batch object have ids from null up to but
       * excluding the result of this function.
       */
      size_t GetNumCalls (void);
      
      /**
       * Get the data for the job with a specific id.
       */
      void GetJobData
         (unsigned                             uJobID,
          unsigned                            &uController,
          CCapiStateMachine::ConnectionType_t &connType,
          std::string                         &strSrcFileName,
          std::string                         &strDstPhoneNumber,
          std::string                         &strSrcPhoneNumber);
   
   
   
   protected:
   
   
   
   private:
      /// The data maintained for each job.
      struct JobData_t
      {
         unsigned                            uController;
         CCapiStateMachine::ConnectionType_t connType;
         std::string                         strSrcFileName;
         std::string                         strDstPhoneNumber;
         std::string                         strSrcPhoneNumber;
      };
      
      /// The array of maintained jobs.
      std::vector<JobData_t> m_aJobs;
   
   

      /**
       * Read a batch file and construct the job array from it.
       */
      void ReadBatchFile
         (const std::string &strBatchFileName);
      
      /**
       * Evaluate a line of a batch file and create a job entry from it.
       */
      void EvalBatchLine
         (const std::string &strLine);



}; // CBatchAdmin





// === Prototypes of interface functions =================================





#endif // __BATCHADMIN_H
