/**
 * @file capitest.cc
 *
 * CapiTest - Main module, argument parsing and function dispatching.
 *
 * Copyright: 2000-2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: capitest.cc,v 1.12.4.1 2005/05/27 16:28:10 thomas Exp $
 * Project  CAPI for BSD
 * Target   capitest - Test tool for the functionality of CAPI for BSD
 * @date    01.01.2003
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

// system includes
#include <iostream>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <capi20.h>
#include <capi_bsd.h>

// import includes

#define __CAPITEST__

// local includes
#include "config.h"
#include "control.h"
#include "capism.h"





// === public definitions ================================================





// === private definitions ===============================================





// === prototypes for private functions ==================================





// === definition of public functions ====================================





/*
        The main entry point
        ====================
*/

int main
   (int    argc,
    char **argv)
{
   CConfig::TaskToExecute_t task;
   int                      iRes;

   // initialize the configuration data (including program name for output)
   e_oConfig.Init (argv [0]);

   // parse the command line
   task = e_oConfig.EvalCommandLine (argc, argv);
   
   // execute the task requested
   switch (task)
   {
      case CConfig::TASK_USAGE:
         e_oConfig.Usage ();
         iRes = 1;
         break;

      case CConfig::TASK_LIST_CTLR:
         iRes = Ctrl_ListAllControllers (e_oConfig.GetVerboseOutput ());
         break;
         
      case CConfig::TASK_IN_CALLS:
         iRes = Ctrl_HandleIncomingCalls ();
         break;
         
      case CConfig::TASK_OUT_CALLS:
      case CConfig::TASK_REP_OUT_CALLS:
         if (e_oConfig.GetDstPhoneNumber ().length () == 0)
         {
            std::cerr << e_oConfig.GetProgName ()
                      << ": A destination number must be specified for outgoing calls"
                      << std::endl;
            return (2);
         }
         if (e_oConfig.GetConnectionType () == CCapiStateMachine::CT_AUTO)
         {
            std::cerr << e_oConfig.GetProgName ()
                      << ": Connection type \"auto\" is not allowed for outgoing calls"
                      << std::endl;
            return (2);
         }
         if (access (e_oConfig.GetOutputFileName ().c_str (), R_OK) != 0)
         {
            std::cerr << e_oConfig.GetProgName ()
                      << ": Output file \"" << e_oConfig.GetOutputFileName ()
                      << "\" is not readable: "
                      << strerror (errno) << " (" << errno << ")"
                      << std::endl;
            return (3);
         }
         iRes = Ctrl_DoOutgoingCalls (task == CConfig::TASK_REP_OUT_CALLS);
         break;
         
      case CConfig::TASK_BATCH_CALLS:
      case CConfig::TASK_REP_BATCH_CALLS:
         if (access (e_oConfig.GetBatchFileName ().c_str (), R_OK) != 0)
         {
            std::cerr << e_oConfig.GetProgName ()
                      << ": Batch file \"" << e_oConfig.GetBatchFileName ()
                      << "\" is not readable: "
                      << strerror (errno) << " (" << errno << ")"
                      << std::endl;
            return (3);
         }
         iRes = Ctrl_DoBatchCalls (e_oConfig.GetBatchFileName (),
                                   task == CConfig::TASK_REP_BATCH_CALLS);
         break;
         
      case CConfig::TASK_SELF_CONNECT:
      case CConfig::TASK_REP_SELF_CONNECT:
         if (e_oConfig.GetDstPhoneNumber ().length () == 0)
         {
            std::cerr << e_oConfig.GetProgName ()
                      << ": a destination number must be specified for a self connection"
                      << std::endl;
            return (2);
         }
         if (e_oConfig.GetConnectionType () == CCapiStateMachine::CT_AUTO)
         {
            std::cerr << e_oConfig.GetProgName ()
                      << ": Connection type \"auto\" is not allowed for a self connection"
                      << std::endl;
            return (2);
         }
         iRes = Ctrl_SelfConnect (task == CConfig::TASK_REP_SELF_CONNECT);
         break;

      default:
      case CConfig::TASK_UNKNOWN:
      case CConfig::TASK_ERROR:
         // Note: An error message is already printed.
         e_oConfig.Usage ();
         iRes = 2;
         break;
   }

   return (iRes);
} /* main */





/* === definition of private functions =================================== */
