/**
 * @file ctlrinfo.h
 *
 * ControllerInfo - Getting, evaluating and printing out information about a
 * CAPI controller.
 *
 * Copyright: 2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: ctlrinfo.h,v 1.3.4.1 2005/05/27 16:28:12 thomas Exp $
 * Project  CAPI for BSD
 * Target   capitest - Test tool for the functionality of CAPI for BSD
 * @date    20.08.2003
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef CTLRINFO_H
#define CTLRINFO_H

#include <ostream>
#include <capi20.h>
#include <capi_bsd.h>





// === Public declarations ===============================================





// === Class declarations ================================================





/**
 * Class for getting, evaluating and printing out CAPI controller information.
 */
class CControllerInfo
{
   public:
      /**
       * Constructor.
       */
      CControllerInfo (void);
      
      /**
       * Destructor.
       */
      ~CControllerInfo (void);
      
      /**
       * Initialize the information about the controller specified.
       */
      unsigned Init
         (unsigned      uCtlrNum,
          std::ostream &es);
      
      /**
       * Check for completed initialization for a controller.
       */
      bool IsInitialized (void);
      
      /**
       * Print out general information about the stored controller.
       */
      void Print
         (std::ostream &os);
      
      /**
       * Print out detailed information about the stored controller.
       */
      void PrintDetailed
         (std::ostream &os);
   
   

   protected:
   
   
   
   private:
      /// The table to translate controller profile global option into strings.
      static const char *sm_aGlobOptStrings [32];

      /// The table to translate B1 protocol numbers into strings.
      static const char *sm_aB1ProtStrings [32];
      
      /// The table to translate B2 protocol numbers into strings.
      static const char *sm_aB2ProtStrings [32];
      
      /// The table to translate B3 protocol numbers into strings.
      static const char *sm_aB3ProtStrings [32];
      
      /// The controller for which information is gathered. If this field is set
      /// to zero, the remaining class members are not initialized.
      unsigned m_uCtlr;
      
      /// CAPI profile data for the controller.
      CAPIProfileBuffer_t m_profile;
      
      /// Additional driver information for the controller.
      CAPICtlrDriverInfo_t m_drvInfo;
      
      /// Memory for the manufacturer string.
      char m_szManufacturer [64];
      
      /// Memory for the serial number.
      char m_szSerialNumber [8];
      
      /// Major version number of the controller firmware.
      unsigned m_uManuMajor;
      
      /// Minor version number of the controller firmware.
      unsigned m_uManuMinor;
      
      /// Major version number of the controller driver.
      unsigned m_uBsdMajor;
      
      /// Minor version number of the controller driver.
      unsigned m_uBsdMinor;
   
   
   
      /**
       * Get all data about a controller from CAPI.
       */
      unsigned GetCtlrData
         (std::ostream &es);
      
      /**
       * Print out a readable version of the global options.
       */
      void PrintGlobalOptions
         (std::ostream &os);
      
      /**
       * Print out a readable version of the supported B1 protocols.
       */
      void PrintB1ProtocolSupport
         (std::ostream &os);
      
      /**
       * Print out a readable version of the supported B2 protocols.
       */
      void PrintB2ProtocolSupport
         (std::ostream &os);
      
      /**
       * Print out a readable version of the supported B3 protocols.
       */
      void PrintB3ProtocolSupport
         (std::ostream &os);
      
      
      
}; // CControllerInfo





// === Prototypes of interface functions =================================





#endif // CTLRINFO_H
