/**
 * @file bound_pointer.cc
 *
 * BoundPointer - Class for a byte pointer with a limited domain.
 *
 * Copyright: 2002-2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: bound_pointer.cc,v 1.5.4.1 2005/05/27 16:28:13 thomas Exp $
 * Project  CAPI for BSD
 * Target   capitrace - Tracing CAPI calls and messages
 * @date    27.12.2002
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

// System includes

// Import includes

#define __BOUND_POINTER__

// Local includes
#ifndef __BOUND_POINTER_H
#  include "bound_pointer.h"
#endif





// === Private definitions ===============================================





// === Definitions for public declarations ===============================





// === Prototypes for private functions ==================================





// === Implementation of class members ===================================





// --- Class for a byte pointer with a limited domain --------------------

/**
 * Constructor.
 */

CBoundPointer::CBoundPointer
   (u_int8_t *pBase,
    size_t    nLength)
{
   m_pBase        = pBase;
   m_pEnd         = pBase + nLength;
   m_pCurr        = pBase;
   
} // CBoundPointer::CBoundPointer



CBoundPointer::CBoundPointer
   (u_int8_t *pBase,
    u_int8_t *pEnd)
{
   m_pBase        = pBase;
   m_pEnd         = pEnd;
   m_pCurr        = pBase;
   
} // CBoundPointer::CBoundPointer



CBoundPointer::CBoundPointer
   (const CBoundPointer &roToCopy)
{
   m_pBase        = roToCopy.m_pBase;
   m_pEnd         = roToCopy.m_pEnd;
   m_pCurr        = roToCopy.m_pCurr;
   
} // CBoundPointer::CBoundPointer





/**
 * Destructor.
 */

CBoundPointer::~CBoundPointer (void)
{
   // nothing to do
   
} // CBoundPointer::~CBoundPointer





/**
 * Accessing the location pointed to by this object.
 */

u_int8_t * CBoundPointer::operator-> (void)
{
   if (m_pCurr == NULL)
   {
      throw CBoundPointerDomainError ("Current address is NULL");
   }
   if (m_pCurr >= m_pEnd)
   {
      throw CBoundPointerDomainError ("Current address is behind the end");
   }
   return (m_pCurr);
} // CBoundPointer::operator->





/**
 * Assignment operator.
 */

CBoundPointer & CBoundPointer::operator= (const CBoundPointer &roToAssign)
{
   m_pBase        = roToAssign.m_pBase;
   m_pEnd         = roToAssign.m_pEnd;
   m_pCurr        = roToAssign.m_pCurr;
   
   return (*this);
} // CBoundPointer::operator=





/**
 * Pre-Increment operator.
 */

CBoundPointer & CBoundPointer::operator++ (void)
{
   if (m_pCurr >= m_pEnd)
   {
      throw CBoundPointerDomainError ("operator++ left domain");
   }
   m_pCurr++;
   
   return (*this);
} // CBoundPointer::operator++





/**
 * Post-Increment operator.
 */

CBoundPointer CBoundPointer::operator++ (int)
{
   if (m_pCurr >= m_pEnd)
   {
      throw CBoundPointerDomainError ("operator++(int) left domain");
   }

   CBoundPointer oCopy (*this);
   
   m_pCurr++;
      
   return (oCopy);
} // CBoundPointer::operator++





/**
 * Increment by a specific amount.
 */

CBoundPointer & CBoundPointer::operator+= (size_t nInc)
{
   if (m_pCurr + nInc > m_pEnd)
   {
      throw CBoundPointerDomainError ("operator+= left domain");
   }
   m_pCurr += nInc;
   
   return (*this);
} // CBoundPointer::operator+=





/**
 * Pre-Decrement operator.
 */

CBoundPointer & CBoundPointer::operator-- (void)
{
   if (m_pCurr <= m_pBase)
   {
      throw CBoundPointerDomainError ("operator-- left domain");
   }
   m_pCurr--;
   
   return (*this);
} // CBoundPointer::operator--





/**
 * Post-Decrement operator.
 */

CBoundPointer CBoundPointer::operator-- (int)
{
   if (m_pCurr <= m_pBase)
   {
      throw CBoundPointerDomainError ("operator--(int) left domain");
   }

   CBoundPointer oCopy (*this);
   
   m_pCurr--;
      
   return (oCopy);
} // CBoundPointer::operator--





/**
 * Decrement by a specific amount.
 */

CBoundPointer & CBoundPointer::operator-= (size_t nDec)
{
   if ((size_t) (m_pCurr - m_pBase) < nDec)
   {
      throw CBoundPointerDomainError ("operator-= left domain");
   }
   m_pCurr -= nDec;
   
   return (*this);
} // CBoundPointer::operator-=





/**
 * Compute the difference between two BoundPointers of the same domain.
 */

size_t CBoundPointer::operator- (const CBoundPointer &roArg2)
{
   // first check both operands for the same domain
   if (m_pBase != roArg2.m_pBase)
   {
      throw CBoundPointerDomainError
               ("operator- called with different base pointers");
   }
   if (m_pEnd != roArg2.m_pEnd)
   {
      throw CBoundPointerDomainError
               ("operator- called with different end pointers");
   }
   
   return (m_pCurr - roArg2.m_pCurr);
} // CBoundPointer::operator-





/**
 * Query for the remaining bytes until the end position.
 */

size_t CBoundPointer::bytes_remaining (void)
{
   return ((size_t) (m_pEnd - m_pCurr));
} // CBoundPointer::bytes_remaining





/**
 * Query for the current address.
 */

u_int8_t *CBoundPointer::get_pointer (void)
{
   if (m_pCurr == NULL)
   {
      throw CBoundPointerDomainError ("Current pointer is NULL");
   }
   if (m_pCurr >= m_pEnd)
   {
      throw CBoundPointerDomainError ("Current pointer is behind the end");
   }
   return (m_pCurr);
} // CBoundPointer::get_pointer





// === Implementation of public functions ================================





// === Implementation of private functions ===============================
