/**
 * @file bound_pointer.h
 *
 * BoundPointer - Class for a byte pointer with a limited domain.
 *
 * Copyright: 2002-2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: bound_pointer.h,v 1.4.4.1 2005/05/27 16:28:13 thomas Exp $
 * Project  CAPI for BSD
 * Target   capitrace - Tracing CAPI calls and messages
 * @date    27.12.2002
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __BOUND_POINTER_H
#define __BOUND_POINTER_H

#include <sys/param.h>
#include <stdexcept>





// === Public declarations ===============================================





// === Class declarations ================================================





/**
 * The exception thrown by CBoundPointer for leaving the domain.
 */

class CBoundPointerDomainError: public std::domain_error
{
   public:
      /**
       * Constructor.
       */
      CBoundPointerDomainError
         (const std::string &strWhatArg):
         domain_error (strWhatArg)
         { }

}; // CBoundPointerDomainError





/**
 * Class for a byte pointer with a limited domain.
 */

class CBoundPointer
{
   public:
      /**
       * Constructor.
       */
      CBoundPointer
         (u_int8_t *pBase,
          size_t    nLength);
      
      CBoundPointer
         (u_int8_t *pBase,
          u_int8_t *pEnd);
      
      CBoundPointer
         (const CBoundPointer &roToCopy);
         
      /**
       * Destructor.
       */
      ~CBoundPointer (void);
      
      /**
       * Accessing the location pointed to by this object.
       */
      u_int8_t * operator-> (void);

      /**
       * Assignment operator.
       */
      CBoundPointer & operator= (const CBoundPointer &roToAssign);
      
      /**
       * Pre-Increment operator.
       */
      CBoundPointer & operator++ (void);
      
      /**
       * Post-Increment operator.
       */
      CBoundPointer operator++ (int);
      
      /**
       * Increment by a specific amount.
       */
      CBoundPointer & operator+= (size_t nInc);
      
      /**
       * Pre-Decrement operator.
       */
      CBoundPointer & operator-- (void);
      
      /**
       * Post-Decrement operator.
       */
      CBoundPointer operator-- (int);
      
      /**
       * Decrement by a specific amount.
       */
      CBoundPointer & operator-= (size_t nDec);
      
      /**
       * Compute the difference between two BoundPointers of the same domain.
       */
      size_t operator- (const CBoundPointer &roArg2);
      
      /**
       * Query for the remaining bytes until the end position.
       */
      size_t bytes_remaining (void);
      
      /**
       * Query for the current address.
       */
      u_int8_t *get_pointer (void);
   
   

   protected:
      /// The base address of the current objects domain.
      u_int8_t *m_pBase;
      
      /// The address one behind the last byte of the current objects domain.
      u_int8_t *m_pEnd;
      
      /// The current address this object points to.
      u_int8_t *m_pCurr;
      
   
   
   private:
      /// No default constructor supported.
      CBoundPointer (void);
   
   
   
}; // CBoundPointer





// === Prototypes of interface functions =================================





#endif // __BOUND_POINTER_H
