/**
 * @file capitrace.cc
 *
 * CapiTrace - CAPI trace tool in the context of C4B.
 *
 * Copyright: 2002-2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: capitrace.cc,v 1.6.4.1 2005/05/27 16:28:13 thomas Exp $
 * Project  CAPI for BSD
 * Target   capitrace - Tracing CAPI calls and messages
 * @date    14.08.2002
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

// System includes

// Import includes

#define __CAPITRACE__

// Local includes
#ifndef __CONFIG_H
#  include "config.h"
#endif
#ifndef ___TRACE_INTERPRETER_H
#  include "trace_interpreter.h"
#endif
#ifndef __TRACER_H
#  include "tracer.h"
#endif





// === Private definitions ===============================================





// === Definitions for public declarations ===============================





// === Prototypes of private functions ===================================





// === Definitions for class members =====================================





// === Definitions for public functions ==================================





/**
 * Main function for the CAPITrace program.
 */

int main
   (int    iArgc,
    char **papszArgv)
{
   // catch any exception and print an error message if one is thrown
   try
   {
      CConfig::TaskToExecute_t task;

      // initialize the configuration handling object
      e_oConfig.Init (papszArgv [0]);

      // evaluate the command line
      try
      {
         task = e_oConfig.EvalCommandLine (iArgc, papszArgv);
      }
      catch (std::exception &e)
      {
         std::cerr << std::endl;
         std::cerr << e_oConfig.GetProgName ()
                   << ": Error evaluating command line: "
                   << e.what () << "." << std::endl;
         std::cerr << std::endl;
         e_oConfig.Usage ();
         return (3);
      }
      catch (...)
      {
         std::cerr << std::endl;
         std::cerr << e_oConfig.GetProgName ()
                   << ": Unknown error evaluating command line." << std::endl;
         std::cerr << std::endl;
         e_oConfig.Usage ();
         return (3);
      }

      // perform requested operation
      switch (task)
      {
         case CConfig::TASK_USAGE:
            // if help is requested: print usage screen
            e_oConfig.Usage ();
            return (1);

         case CConfig::TASK_INTERPRET_TRACE:
            // if operation is interpreting a formerly written trace file:
            {
               CTraceInterpreter interpreter;

               // perform trace file interpretation
               interpreter.SetInputFile
                  (e_oConfig.GetInterpretSourceFileName ());
               interpreter.SetOutputFile
                  (e_oConfig.GetInterpretOutputFileName ());
               interpreter.SetDataB3MsgOutput
                  (e_oConfig.GetInterpretDataB3Msgs (),
                   e_oConfig.GetInterpretDataB3Blocks (),
                   e_oConfig.GetInterpretDataB3BlockLen ());
               interpreter.SetHexOutput (e_oConfig.GetInterpretHexOutput ());
               interpreter.Run ();
            }
            break;

         case CConfig::TASK_START_TRACER:
            // if operation is to start a tracer (application or controller):
            {
               CTracerList                                tracerList;
               CConfig::CTracerConfigList::const_iterator iter;

               // loop until all command line parameters are handled
               for (iter = e_oConfig.GetTracerConfigStartIterator ();
                    iter != e_oConfig.GetTracerConfigEndIterator ();
                    ++iter)
               {
                  // create a tracer object with the current parameters for
                  // later start
                  tracerList.Add (*iter);
               }

               // now start all tracer objects created and save their output
               tracerList.Run ();
            }
            break;

         default:
            // unknown operation or error in command line evaluation
            e_oConfig.Usage ();
            return (2);
      }
   }
   catch (std::exception &e)
   {
      std::cerr << std::endl;
      std::cerr << e_oConfig.GetProgName () << ": Execution error: "
                << e.what () << "." << std::endl;
      std::cerr << std::endl;
      return (3);
   }
   catch (...)
   {
      std::cerr << std::endl;
      std::cerr << e_oConfig.GetProgName ()
                << ": Unknown execution error." << std::endl;
      std::cerr << std::endl;
      return (3);
   }
   
   return (0);
} // main





// === Definitions for private functions =================================
