/**
 * @file trace_message_print.h
 *
 * TraceMessagePrint - Class to convert CAPI trace message to a textual
 * representation.
 *
 * Copyright: 2002-2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: trace_message_print.h,v 1.8.2.1 2005/05/27 16:28:15 thomas Exp $
 * Project  CAPI for BSD
 * Target   capitrace - Tracing CAPI calls and messages
 * @date    03.12.2002
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __TRACE_MESSAGE_PRINT_H
#define __TRACE_MESSAGE_PRINT_H

#include <stddef.h> // size_t
#include <capi_bsdtrc.h>

#ifndef __BOUND_POINTER_H
#  include "bound_pointer.h"
#endif
#ifndef __FORM_OUT_STREAM_H
#  include "form_out_stream.h"
#endif





// === Public declarations ===============================================





// === Class declarations ================================================





/**
 * Class to convert CAPI trace messages to a textual representation.
 */
class CTraceMessagePrint
{
   public:
      /**
       * Constructor.
       */
      CTraceMessagePrint
         (CFormOutStream &ros);
      
      /**
       * Destructor.
       */
      ~CTraceMessagePrint (void);
      
      /**
       * Print a CAPI trace message to the output stream member.
       *
       * Assumption: The buffer specified is large enough to hold at least the
       *             trace message header.
       *
       * @param pTraceMsg       I: The address of the message to interpret.
       * @param nLenTraceMsg    I: The length of the message in bytes for error
       *                           checking.
       *
       * @return Nothing.
       */
      void Print
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
   
   
   
   protected:
      /// The entry type for the following global dispatch table.
      struct MsgTypeDispatchTableEntry_t
      {
         unsigned uMsgType;
         void (CTraceMessagePrint::*pfnPrintMsg)
            (const CAPITraceMsg_t *pTraceMsg,
             size_t                nLenTraceMsg);
      };
      
      /// The global dispatch table to print a message of a specific type.
      static MsgTypeDispatchTableEntry_t m_aMsgTypeDispTbl [];
   
      /// The entry type for the list of fields to print for a CAPI message.
      struct CapiMsgFieldDescriptionEntry_t
      {
         const char *pszPrompt;
         void (CTraceMessagePrint::*pfnPrintField)
            (CBoundPointer &addr,
             const char    *pszPrompt,
             unsigned       uValueIndent);
      };
      
      /// The list of fields for a Connect-Request.
      static CapiMsgFieldDescriptionEntry_t m_aConnectReqList [];
      
      /// The list of fields for a Connect-Indication.
      static CapiMsgFieldDescriptionEntry_t m_aConnectIndList [];
      
      /// The list of fields for a Connect-Response.
      static CapiMsgFieldDescriptionEntry_t m_aConnectRespList [];
      
      /// The list of fields for a Connect-Active-Indication.
      static CapiMsgFieldDescriptionEntry_t m_aConnActIndList [];
      
      /// The list of fields for a Disconnect-Indication.
      static CapiMsgFieldDescriptionEntry_t m_aDisconnectIndList [];
      
      /// The list of fields for a Listen-Request.
      static CapiMsgFieldDescriptionEntry_t m_aListenReqList [];
      
      /// The list of fields for a Info-Request.
      static CapiMsgFieldDescriptionEntry_t m_aInfoReqList [];
      
      /// The list of fields for a Info-Indication.
      static CapiMsgFieldDescriptionEntry_t m_aInfoIndList [];
      
      /// The list of fields for a Select-B-Protocol-Request.
      static CapiMsgFieldDescriptionEntry_t m_aSelBProtReqList [];
      
      /// The list of fields for a Facility-Request.
      static CapiMsgFieldDescriptionEntry_t m_aFacilityReqList [];

      /// The list of fields for a Facility-Confirm.
      static CapiMsgFieldDescriptionEntry_t m_aFacilityConfList [];

      /// The list of fields for a Facility-Indication.
      static CapiMsgFieldDescriptionEntry_t m_aFacilityIndList [];

      /// The list of fields for a Facility-Response.
      static CapiMsgFieldDescriptionEntry_t m_aFacilityRespList [];

      /// The list of fields for a Connect-B3-Response.
      static CapiMsgFieldDescriptionEntry_t m_aConnectB3RespList [];
      
      /// The list of fields for a Disconnect-B3-Indication.
      static CapiMsgFieldDescriptionEntry_t m_aDisconnectB3IndList [];
      
      /// The list of fields for a Data-B3-Request.
      static CapiMsgFieldDescriptionEntry_t m_aDataB3ReqList [];
      
      /// The list of fields for a Data-B3-Confirm.
      static CapiMsgFieldDescriptionEntry_t m_aDataB3ConfList [];
      
      /// The list of fields for a Data-B3-Indication.
      static CapiMsgFieldDescriptionEntry_t m_aDataB3IndList [];
      
      /// The list of fields for a Data-B3-Response.
      static CapiMsgFieldDescriptionEntry_t m_aDataB3RespList [];
      
      /// The list of fields for a message with only Additional Info.
      static CapiMsgFieldDescriptionEntry_t m_aAddInfoList [];
      
      /// The list of fields for a message with only an Info field.
      static CapiMsgFieldDescriptionEntry_t m_aInfoList [];
      
      /// The list of fields for a message with only a NCPI.
      static CapiMsgFieldDescriptionEntry_t m_aNcpiList [];
      
      /// The list of fields for Manufacturer messages.
      static CapiMsgFieldDescriptionEntry_t m_aManuMsgList [];
      
      /// The entry type for the following dispatch table for CAPI messages.
      struct CapiMsgDispatchTableEntry_t
      {
         u_int16_t                       wMsgCmd;
         CapiMsgFieldDescriptionEntry_t *paMsgFields;
         size_t                          nNumMsgFields;
      };
      
      /// The global table to assign a field description list to every CAPI
      /// message.
      static CapiMsgDispatchTableEntry_t m_aCapiMsgDispTbl [];
      
      /// The reference to the output stream to send any interpreted CAPI trace
      /// message to.
      CFormOutStream &m_ros;
      
      /// The indentation of trace messages in the output.
      static const size_t m_nNormalIndent;
      
   
   
      /**
       * Print an application register message.
       */
      void PrintApplRegister
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application release message.
       */
      void PrintApplRelease
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application put message call.
       */
      void PrintApplPutMessage
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application get message call.
       */
      void PrintApplGetMessage
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application set signal call.
       */
      void PrintApplSetSignal
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application get manufacturer call.
       */
      void PrintApplGetManufacturer
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application get version call.
       */
      void PrintApplGetVersion
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application get serial number call.
       */
      void PrintApplGetSerialNumber
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application get profile call.
       */
      void PrintApplGetProfile
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application get controller driver info call.
       */
      void PrintApplGetCtlrDriverInfo
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application reset controller call.
       */
      void PrintApplResetCtlr
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an application CAPI message.
       */
      void PrintApplMsg
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print a driver level application register message.
       */
      void PrintDrvrRegister
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print a driver level application release message.
       */
      void PrintDrvrRelease
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print a driver level put message call.
       */
      void PrintDrvrPutMessage
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print a driver level get message call.
       */
      void PrintDrvrGetMessage
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print a driver level reset controller call.
       */
      void PrintDrvrResetCtlr
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print a driver level CAPI message.
       */
      void PrintDrvrMsg
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);
      
      /**
       * Print an unknown message
       */
      void PrintUnknownMessage
         (const CAPITraceMsg_t *pTraceMsg,
          size_t                nLenTraceMsg);

      /**
       * Print a general CAPI message.
       */
      void PrintCapiMsg
         (const CAPIMsg_t *pCapiMsg,
          size_t           nLenCapiMsg,
          u_int8_t        *pData,
          size_t           nLenData);
      
      /**
       * Print the info fields of a CAPI message.
       */
      void PrintCapiMsgInfoFields
         (CBoundPointer                  &addr,
          CapiMsgFieldDescriptionEntry_t *paMsgFields,
          size_t                          nNumMsgFields);
      
      /**
       * Print a string representation of a CIP value.
       */
      void PrintCipValue
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);

      /**
       * Print a Called Party Number.
       */
      void PrintCalledPartyNumber
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);

      /**
       * Print a Calling Party Number.
       */
      void PrintCallingPartyNumber
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);

      /**
       * Print B-Protocol information.
       */
      void PrintBProtocol
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print B1-Configuration.
       */
      void PrintB1Configuration
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uB1Protocol,
          unsigned       uValueIndent);
      
      /**
       * Print B2-Configuration.
       */
      void PrintB2Configuration
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uB2Protocol,
          unsigned       uValueIndent);
      
      /**
       * Print B3-Configuration.
       */
      void PrintB3Configuration
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uB3Protocol,
          unsigned       uValueIndent);
      
      /**
       * Print B-Protocol Global Configuration structure.
       */
      void PrintBProtGlobalConfig
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print an Additional Info structure.
       */
      void PrintAdditionalInfo
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);

      /**
       * Print a B-channel Information structure.
       */
      void PrintBChannelInfo
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);

      /**
       * Print a Sending Complete structure.
       */
      void PrintSendingComplete
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);

      /**
       * Print a string representation of the flags in a Data-B3 message.
       */
      void PrintDataB3Flags
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a string representation of a Facility Selector.
       */
      void PrintFacilitySelector
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print the meaning of an Info Number in an Info Indication.
       */
      void PrintInfoNumber
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a reject value.
       */
      void PrintRejectValue
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a CAPI result / info / reason value.
       */
      void PrintResultValue
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a QWord value in hexadecimal.
       */
      void PrintHexQWord
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a DWord value in hexadecimal.
       */
      void PrintHexDWord
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a Word value in hexadecimal.
       */
      void PrintHexWord
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a Word value in decimal.
       */
      void PrintDecWord
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a CAPI struct as a string.
       */
      void PrintStringStruct
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Print a CAPI struct as a byte sequence.
       */
      void PrintCapiStruct
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);

      /**
       * Print the rest of a message as a byte sequence.
       */
      void PrintRestAsByteSequence
         (CBoundPointer &addr,
          const char    *pszPrompt,
          unsigned       uValueIndent);
      
      /**
       * Translate a CAPI message command into a string.
       */
      const char *GetCapiMsgName
         (unsigned uCmd);

      /**
       * Translate a B1-Protocol value into a string.
       */
      const char *GetB1ProtocolName
         (unsigned uB1Protocol);
      
      /**
       * Translate a B2-Protocol value into a string.
       */
      const char *GetB2ProtocolName
         (unsigned uB2Protocol);
      
      /**
       * Translate a B3-Protocol value into a string.
       */
      const char *GetB3ProtocolName
         (unsigned uB3Protocol);



   private:
   
   
   
}; // CTraceMessagePrint





// === Prototypes of interface functions =================================





#endif // __TRACE_MESSAGE_PRINT_H
