/**
 * @file ctlrlist.h
 *
 * ControllerList - Class for maintaining a list of CAPI controllers.
 *
 * Copyright: 2005 Thomas Wintergerst. All rights reserved.
 *
 * $Id: ctlrlist.h,v 1.1.2.1 2005/05/27 16:28:18 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     Common source files for c4b tools $
 * @date        06.05.2005
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef CTLRLIST_H
#define CTLRLIST_H

#include <string>
#include <list>
#include <capi_bsd.h>





// === Public declarations ===============================================





// === Class declarations ================================================





/**
 * The data maintained for a controller of the list.
 */
class CController
{
   public:
      /**
       * Constructor.
       */
      CController
         (unsigned                    uCtlrNum,
          const CAPICtlrDriverInfo_t &driverInfo);
   
      /**
       * Get the name of the controller.
       */
      const std::string &GetControllerName (void) const;
      
      /**
       * Get the name of the board the controller is on.
       *
       * The board name in this context is defined as the first part of the
       * controller name delimited by a "/" character. If there is none the
       * board name and the controller name are identical.
       *
       * The "/" character is mostly used to add a port number to the name of
       * multi-port boards, so the controller name is unique. This port number
       * suffix is substracted when determining the board name.
       */
      const std::string &GetBoardName (void) const;
      
      /**
       * Get the CAPI controller number.
       */
      unsigned GetControllerNumber (void) const;
   
      /**
       * Get the port index for this controller.
       */
      unsigned GetPortIndex (void) const;
      
      /**
       * Ask if the name of the controller is identical to a string.
       */
      bool NameIs
         (const char *psz);

      /**
       * Ask if the name of the controller's board is identical to a string.
       */
      bool BoardNameIs
         (const char *psz);

      /**
       * Ask if this controller's number is identical to a specific number.
       */
      bool CtlrNumIs
         (unsigned u);
      
   
   
   protected:
   
   
   
   private:
      /// The CAPI controller number.
      unsigned m_uCtlrNum;
      
      /// The information retrieved from CAPI for this controller.
      CAPICtlrDriverInfo_t m_ctlrInfo;
      
      /// The controller name to be returned as a constant string reference.
      std::string m_strName;
      
      /// The board name the controller is on.
      std::string m_strBoardName;
      
      
      
}; // CController



/**
 * Class for maintaining a list of CAPI controllers.
 */
class CControllerList: public std::list<CController>
{
   public:
      /**
       * Constructor.
       *
       * @param strDriverName   I: The name of the controller driver to look
       *                           for. Only controllers for this driver name
       *                           are inserted into the list.
       * @param fQuiet          I: Print out a line of all controllers added to
       *                           the list or not.
       *
       * @return Nothing.
       */
      CControllerList
         (const std::string &strDriverName,
          bool               fQuiet);

      /**
       * Destructor.
       */
      ~CControllerList (void);
      
      /**
       * Find a controller through its name.
       */
      const_iterator find
         (const std::string &strName);

      /**
       * Find a controller through its CAPI controller number.
       */
      const_iterator find
         (unsigned uCtlrNum);
      
      /**
       * Find a controller through its board name.
       */
      const_iterator find_by_board_name
         (const std::string &strBoardName);
      
   
   
   protected:
   
   

   private:
   
   
   
}; // CControllerList





// === Prototypes of interface functions =================================





#endif // CTLRLIST_H
