/**
 * @file daic_globalid.h
 *
 * Daic-GlobalId - Module to handle the global id (D-channel access) state
 * machine.
 *
 * Copyright: 2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: daic_globalid.h,v 1.5.2.1 2005/05/27 16:28:28 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     daic - CAPI manager driver for Diehl active ISDN controllers $
 * @date        12.04.2003
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __DAIC_GLOBALID_H
#define __DAIC_GLOBALID_H

#ifndef __DAIC_GLOBAL_H
#  include <c4b/driver/daic/daic_global.h>
#endif
#ifndef __DAIC_HW_H
#  include <c4b/driver/daic/daic_hw.h>
#endif





/* === Public declarations =============================================== */





/* === Prototypes of interface functions ================================= */





/**
 * Reset the state machine for the global id (D-channel access) of a port.
 *
 * @param pPortData             I/O: The address for the port data for this
 *                                 operation.
 *
 * @return Nothing.
 */
extern void daicgid_reset
   (DaicPortData_t *pPortData);

/**
 * Start the state machine for the global id (D-channel access).
 *
 * The global id state machine must be in state Idle before calling this
 * function. The call will initiate the allocation of a new signalling instance
 * to be informed about new incoming calls. This must be done for each port of
 * an ISDN board. But this function will only work on the specific port given as
 * the argument to this function.
 *
 * @param pPortData             I/O: The port data for this operation.
 *
 * @return CAPI result value, CAPI_OK on success.
 */
extern unsigned daicgid_start
   (DaicPortData_t *pPortData);

/**
 * Handle the return code for a global id (D-channel access) assign request.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param pRcData               I: The return code data to evaluate.
 *
 * @return Nothing.
 */
extern void daicgid_handle_assign_rc
   (DaicPortData_t     *pPortData,
    const DaicRcData_t *pRcData);

/**
 * Handle a return code for a global id (D-channel access) regular request.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param pRcData               I: The return code data to evaluate.
 *
 * @return Nothing.
 */
extern void daicgid_handle_rc
   (DaicPortData_t     *pPortData,
    const DaicRcData_t *pRcData);

/**
 * Handle an indication for the global id (D-channel access).
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param pIndData              I: The indication data to evaluate.
 *
 * @return Nothing.
 */
extern void daicgid_handle_ind
   (DaicPortData_t      *pPortData,
    const DaicIndData_t *pIndData);

/**
 * Handle a CAPI message for the global id (D-channel access).
 *
 * The only messages, that are directly addressed to a controller are a
 * Listen-Request and an Info-Request. An Info-Request to a controller is not
 * supported by this driver. All other messages will be rejected.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uApplID               I: The application id the CAPI message is from.
 * @param pmbMsg                I: The CAPI message to evaluate.
 *
 * @return CAPI result value, CAPI_OK on success.
 */
extern unsigned daicgid_handle_capi_msg
   (DaicPortData_t  *pPortData,
    unsigned         uApplID,
    struct mbuf     *pmbMsg);





#endif /* __DAIC_GLOBALID_H */
