/**
 * @file daic_misc.h
 *
 * Daic-Misc - Miscellaneous helper functions for the daic device driver.
 *
 * Copyright: 2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: daic_misc.h,v 1.8.2.1 2005/05/27 16:28:30 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     daic - CAPI manager driver for Diehl active ISDN controllers $
 * @date        07.04.2003
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __DAIC_MISC_H
#define __DAIC_MISC_H

#ifndef __DAIC_GLOBAL_H
#  include <c4b/driver/daic/daic_global.h>
#endif





/* === Public declarations =============================================== */





/* --- some strings for naming synchronization objects --- */

/** The name of the access mutex for a controller. */
#define DAICMISC_ACCESS_MUTEX_MSG       "daic access mutex"

/** The name of the condition object for signalling of events. */
#define DAICMISC_NOTIFY_CONDITION_MSG   "daic notify condition"



/* --- some strings for the tsleep call --- */

/** The string to display while waiting for exclusive controller access. */
#define DAICMISC_WAIT_FOR_CTLR_ACCESS_MSG       "wait for ctlr access"

/** The string to display while waiting for on-board memory check. */
#define DAICMISC_WAIT_FOR_MEMCHECK              "wait for on-board memory check"

/**
 * The string to display while waiting for the board to accept a firmware
 * portion.
 */
#define DAICMISC_WAIT_FOR_FIRMWARE_DATA         "wait for firmware portion ready"

/**
 * The string to display while waiting for the firmware to acknowledge
 * successful start.
 */
#define DAICMISC_WAIT_FOR_FIRMWARE_START        "wait for firmware start"

/** The string to display while waiting for a test interrupt while downloading. */
#define DAICMISC_WAIT_FOR_READY_INT             "wait for test interrupt"



/* --- some helper macros --- */

#define ARRAY_COUNT( _a )       (sizeof (_a) / sizeof ((_a) [0]))





/* === Prototypes of interface functions ================================= */





/* --- serializing access to a controller --- */

/**
 * Get access to a controller for an operation.
 *
 * @note This function is called by all CAPI manager entry functions. The
 *       interrupt routine must not call it, but must directly obtain the board
 *       access mutex.
 *
 * @param pSc                   I/O: The softc structure for the board.
 *
 * @retval 0                    Unable to obtain board access, timeout.
 * @retval 1                    Exclusive board access gained.
 */
extern int daicmisc_get_ctlr_access
   (DaicSc_t *pSc);

/**
 * Release access to a controller after an operation.
 */
extern void daicmisc_release_ctlr_access
   (DaicSc_t *pSc);



/* --- some helper functions --- */

/**
 * Translate a card type into a string.
 */
extern const char *daicmisc_get_card_type_name
   (DaicCardType_t cardType);

/**
 * Translate a controller return code into a CAPI result value.
 *
 * @param bRc                   I: Controller return code to translate.
 * @param iQualify              I: The originating layer that the return code
 *                                 occurred in. One of DAIC_ID_QUALIFY_*.
 *
 * @return The translated CAPI result value.
 */
extern unsigned daicmisc_get_capi_info_from_rc
   (u_int8_t bRc,
    int      iQualify);

/**
 * Translate a CAPI reject value into a Q.850 cause value.
 */
extern unsigned daicmisc_get_cause_from_capi_reject
   (unsigned uRejectVal);





#endif /* __DAIC_MISC_H */
