/**
 * @file daic_ncci.h
 *
 * Daic-NCCI - The NCCI state machine for the daic device driver.
 *
 * Copyright: 2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: daic_ncci.h,v 1.11.2.1 2005/05/27 16:28:31 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     daic - CAPI manager driver for Diehl active ISDN controllers $
 * @date        02.02.2003
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __DAIC_NCCI_H
#define __DAIC_NCCI_H

#ifndef __DAIC_GLOBAL_H
#  include <c4b/driver/daic/daic_global.h>
#endif
#ifndef __DAIC_HW_H
#  include <c4b/driver/daic/daic_hw.h>
#endif





/* === Public declarations =============================================== */





/* === Prototypes of interface functions ================================= */





/**
 * Reset the NCCI state machines of all NCCIs for a port.
 *
 * @param pPortData             I/O: The address for the port data for this
 *                                 operation.
 *
 * @return Nothing.
 */
extern void daicncci_reset_all
   (DaicPortData_t *pPortData);

/**
 * Assign a free NCCI entry for a new logical connection.
 *
 * @param pPortData             I/O: the port data as the operation context.
 * @param pPlciData             I/O: The data for the PLCI, the new NCCI will be
 *                                 assigned for.
 * @param uChId                 I: The newly assigned channel id for the new
 *                                 NCCI.
 *
 * @retval NULL                 No more NCCI available.
 * @retval Else                 The address of the NCCI data for the newly
 *                              assigned NCCI.
 */
extern DaicNcciData_t *daicncci_alloc_new_ncci
   (DaicPortData_t *pPortData,
    DaicPlciData_t *pPlciData,
    unsigned        uChId);

/**
 * Release all connections assigned to an application id.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uApplID               I: The appl. id for the release operation.
 *
 * @return Nothing.
 */
extern void daicncci_release_appl
   (DaicPortData_t *pPortData,
    unsigned        uApplID);

/**
 * Handle the result of a regular request (no assign).
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uNcci                 I: The NCCI value the return code shall be
 *                                 dispatched to.
 * @param pRcData               I: The return code data to evaluate.
 *
 * @return Nothing.
 */
extern void daicncci_handle_rc
   (DaicPortData_t     *pPortData,
    unsigned            uNcci,
    const DaicRcData_t *pRcData);

/**
 * Enqueue an indication from the controller for a NCCI.
 *
 * If there is currently no hardware request pending and the message queue for
 * the NCCI is empty, the indication will be handled at once.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uNcci                 I: The NCCI value the return code shall be
 *                                 dispatched to.
 * @param pmbIndData            I/O: The indication data to evaluate. The
 *                                 ownership of the mbuf changes to the called
 *                                 function.
 *
 * @return Nothing.
 */
extern void daicncci_enqueue_ind
   (DaicPortData_t *pPortData,
    unsigned        uNcci,
    struct mbuf    *pIndData);

/**
 * Enqueue a CAPI message to a NCCI instance.
 *
 * If there is currently no hardware request pending and the message queue for
 * the NCCI is empty, the message will be handled at once.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uNcci                 I: The addressed NCCI value, equal to the
 *                                 network id subtracted by the network global
 *                                 id.
 * @param uApplID               I: The application id the CAPI message is from.
 * @param pmbMsg                I: The CAPI message to evaluate.
 *
 * @return CAPI result value, CAPI_OK on success.
 */
extern unsigned daicncci_enqueue_capi_msg
   (DaicPortData_t  *pPortData,
    unsigned         uNcci,
    unsigned         uApplID,
    struct mbuf     *pmbMsg);

/**
 * Start removing all NCCIs for a disconnected PLCI.
 *
 * For each removed NCCI the PLCI layer is informed.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uPlci                 I: The disconnected PLCI for removing all
 *                                 associated NCCIs.
 * @param uReason               I: The reason to signal for each NCCI to be
 *                                 removed.
 *
 * @return Nothing.
 */
extern void daicncci_remove_all_ncci_for_plci
   (DaicPortData_t *pPortData,
    unsigned        uPlci,
    unsigned        uReason);





#endif /* __DAIC_NCCI_H */
