/**
 * @file daic_plci.h
 *
 * Daic-PLCI - The PLCI state machine for the daic device driver.
 *
 * Copyright: 2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: daic_plci.h,v 1.12.2.1 2005/05/27 16:28:33 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     daic - CAPI manager driver for Diehl active ISDN controllers $
 * @date        02.02.2003
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __DAIC_PLCI_H
#define __DAIC_PLCI_H

#ifndef __DAIC_GLOBAL_H
#  include <c4b/driver/daic/daic_global.h>
#endif
#ifndef __DAIC_HW_H
#  include <c4b/driver/daic/daic_hw.h>
#endif





/* === Public declarations =============================================== */





/* === Prototypes of interface functions ================================= */





/**
 * Reset the PLCI state machines of all PLCIs for a port.
 *
 * @param pPortData             I/O: The address for the port data for this
 *                                 operation.
 *
 * @return Nothing.
 */
extern void daicplci_reset_all
   (DaicPortData_t *pPortData);

/**
 * Assign a free PLCI entry for a new logical connection.
 *
 * @param pPortData             I/O: the port data as the operation context.
 * @param uSigId                I: The signaling id for the connection if known.
 *                                 This will only be the case for incoming
 *                                 connections, where the signaling id is taken
 *                                 from the global id state machine. For
 *                                 outgoing connections, originating from a
 *                                 CAPI Connect-Request, a signaling id must
 *                                 still be assigned. So in this case it is
 *                                 simply specified as 0.
 *
 * @retval NULL                 No more PLCI available.
 * @retval Else                 The address of the PLCI data for the newly
 *                              assigned PLCI.
 */
extern DaicPlciData_t *daicplci_alloc_new_plci
   (DaicPortData_t *pPortData,
    unsigned        uSigId);

/**
 * Release all connections assigned to an application id.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uApplID               I: The appl. id for the release operation.
 * @param iApplIdx              I: The index into the application data array for
 *                                 the application id.
 *
 * @return Nothing.
 */
extern void daicplci_release_appl
   (DaicPortData_t *pPortData,
    unsigned        uApplID,
    int             iApplIdx);

/**
 * Handle a new incoming call.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param pPlciData             I/O: The PLCI data for this operation.
 * @param pIndData              I: The indication data to evaluate.
 *
 * @retval CAPI_OK              The message was handled successfully, the PLCI
 *                              state machine for the addressed PLCI is running.
 * @retval CDISC_CALL_TO_ANOTHER_APPLICATION
 *                              There is no application registered that can
 *                              handle the new call.
 * @retval Else                 Error occurred.
 */
extern unsigned daicplci_handle_indicate_ind
   (DaicPortData_t      *pPortData,
    DaicPlciData_t      *pPlciData,
    const DaicIndData_t *pIndData);

/**
 * Handle the result of an assign request to obtain a new signaling id.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uPlci                 I: The PLCI value the return code shall be
 *                                 dispatched to.
 * @param pRcData               I: The return code data to evaluate.
 *
 * @return Nothing.
 */
extern void daicplci_handle_sig_assign_rc
   (DaicPortData_t     *pPortData,
    unsigned            uPlci,
    const DaicRcData_t *pRcData);

/**
 * Handle the result of an assign request to obtain a new network id.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uPlci                 I: The PLCI value the return code shall be
 *                                 dispatched to.
 * @param pRcData               I: The return code data to evaluate.
 *
 * @return Nothing.
 */
extern void daicplci_handle_net_assign_rc
   (DaicPortData_t     *pPortData,
    unsigned            uPlci,
    const DaicRcData_t *pRcData);

/**
 * Handle the result of a regular request (no assign).
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uPlci                 I: The PLCI value the return code shall be
 *                                 dispatched to.
 * @param pRcData               I: The return code data to evaluate.
 *
 * @return Nothing.
 */
extern void daicplci_handle_rc
   (DaicPortData_t     *pPortData,
    unsigned            uPlci,
    const DaicRcData_t *pRcData);

/**
 * Enqueue an indication from the controller for a PLCI.
 *
 * If there is currently no hardware request pending and the message queue for
 * the PLCI is empty, the indication will be handled at once.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uPlci                 I: The PLCI value the indication shall be
 *                                 dispatched to.
 * @param pmbIndData            I/O: The indication data to evaluate. The
 *                                 ownership of the mbuf changes to the called
 *                                 function.
 *
 * @return Nothing.
 */
extern void daicplci_enqueue_ind
   (DaicPortData_t *pPortData,
    unsigned        uPlci,
    struct mbuf    *pmbIndData);

/**
 * Enqueue a CAPI message to a PLCI instance.
 *
 * If there is currently no hardware request pending and the message queue for
 * the PLCI is empty, the message will be handled at once.
 *
 * @param pPortData             I/O: The port data for this operation.
 * @param uPlci                 I: The addressed PLCI value, equal to the
 *                                 signalling id.
 * @param uApplID               I: The application id the CAPI message is from.
 * @param pmbMsg                I: The CAPI message to evaluate.
 *
 * @return CAPI result value, CAPI_OK on success.
 */
extern unsigned daicplci_enqueue_capi_msg
   (DaicPortData_t  *pPortData,
    unsigned         uPlci,
    unsigned         uApplID,
    struct mbuf     *pmbMsg);

/**
 * Count the release of an NCCI associated with a PLCI.
 *
 * @param pPortData             I/O: The port data as the operation context.
 * @param uPlci                 I: The PLCI, the NCCI is released for.
 *
 * @return Nothing.
 */
extern void daicplci_ncci_removed
   (DaicPortData_t *pPortData,
    unsigned        uPlci);





#endif /* __DAIC_PLCI_H */
