/**
 * @file ix1a_bpci.h
 *
 * IX1a-BasicPCI - Routines and specifications for IX1 ISA boards.
 *
 * Copyright: 2004 Thomas Wintergerst. All rights reserved.
 *
 * $Id: ix1a_bpci.h,v 1.3.2.1 2005/05/27 16:28:35 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     ix1a - CAPI manager driver for IX1 active ISDN controllers $
 * @date        14.12.2004
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __IX1A_BPCI_H
#define __IX1A_BPCI_H





/* === Public declarations =============================================== */





/* === Prototypes of interface functions ================================= */





/**
 *  Initialise and fill the resource info structure for a Basic PCI board.
 *
 * This function will initialise the resource info structure for Basic PCI
 * boards and acquire the needed resources.
 *
 * @param dev                   I: The device entry for the board to register.
 * @param pResInfo              I/O: The resource info structure to fill.
 *
 * @retval 0                    All resources could be acquired.
 * @retval Else                 At least one resource could not be acquired.
 *                              In fact no resource is required in this case.
 */
extern int ix1abpci_init_resinfo
   (device_t            dev,
    Ix1aResourceInfo_t *pResInfo);

/**
 * Initialise Basic PCI board for operation.
 *
 * @param dev                   I: The device representing the board.
 * @param pSc                   I/O: The softc structure for the board.
 *
 * @retval 0                    The tests were successful, board exists and is
 *                              working.
 * @retval Else                 Error occurred, board does not exist or is not
 *                              working.
 */
extern int ix1abpci_init_board
   (device_t  dev,
    Ix1aSc_t *pSc);

/**
 * Enable a Basic PCI board after disabling it.
 *
 * @note This routine expects the board to have been correctly initialised
 *       before disabling.
 *
 * @param pSc                   I/O: The softc structure for the board to
 *                                 enable.
 *
 * @retval 0                    The board was successfully enabled.
 * @retval Else                 Enabling the board failed, the result is an
 *                              errno value.
 */
extern int ix1abpci_enable_board
   (Ix1aSc_t *pSc);

/**
 * Disable a Basic PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board to
 *                                 disable.
 *
 * @return Nothing.
 */
extern void ix1abpci_disable_board
   (Ix1aSc_t *pSc);

/**
 * Set the offset pointer within a Basic PCI board's shared memory.
 *
 * For Basic PCI boards this seems to be a non-reliant operation, according to
 * the original Linux and DOS driver sources from ITK / Digi Europe. Before and
 * after writing to the offset register one or two i/o read operations seem to
 * be necessary and the write operation to the offset register may even fail
 * for some attempts. So we implement this operation with a loop that checks
 * the result of the operation and performs a number of attempts to set the
 * offset register.
 *
 * @param pSc                   I/O: The softc structure for the board to set
 *                                 the offset register.
 * @param uAddr                 I: The address offset to set.
 *
 * @retval 0                    The offset register was set successfully.
 * @retval Else                 The offset register could not be set, board
 *                              failure.
 */
extern int ix1abpci_shm_set_offset
   (Ix1aSc_t *pSc,
    unsigned  uAddr);

/**
 * Read a 16bit word from a shared memory address of a Basic PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to read from.
 * @param puValue               O: The word read from the board, only valid if
 *                                 the result is 0.
 *
 * @retval 0                    The value was read successfully and stored into
 *                              *puValue.
 * @retval Else                 An errno value on failure.
 */
extern int ix1abpci_shm_read_word
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    unsigned *puValue);

/**
 * Read a block of data from a shared memory address of a Basic PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to read from.
 * @param pbData                I: The address for returing the data read.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 read.
 *
 * @retval 0                    The data block was read successfully.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1abpci_shm_read_block_16
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    u_int8_t *pbData,
    size_t    nLenData);

/**
 * Write a 16bit word to a shared memory address of a Basic PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param uValue                I: The word to write.
 *
 * @retval 0                    The value was successfully written.
 * @retval Else                 An errno value on failure.
 */
extern int ix1abpci_shm_write_word
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    unsigned  uValue);

/**
 * Write a block of data to a shared memory address of a Basic PCI board using
 * 16bit mode to write every single byte.
 *
 * This function is called for boot code download to write the boot code. Every
 * single byte must be written using 16bit i/o, else the shared memory pointer
 * would not increment for the Basic PCI.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param pbData                I: The address of the data block to write.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 write.
 *
 * @retval 0                    The data block was written successfully.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1abpci_shm_write_block_8
   (Ix1aSc_t       *pSc,
    unsigned        uAddr,
    const u_int8_t *pbData,
    size_t          nLenData);

/**
 * Write a block of data to a shared memory address of a Basic PCI board using
 * 16bit mode.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param pbData                I: The address of the data block to write.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 write.
 *
 * @retval 0                    The data block was written successfully.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1abpci_shm_write_block_16
   (Ix1aSc_t       *pSc,
    unsigned        uAddr,
    const u_int8_t *pbData,
    size_t          nLenData);

/**
 * Compare a block of data to a shared memory address of a Basic PCI board
 * using 16bit mode.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param pbData                I: The address of the data block to write.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 write.
 * @param pfResult              O: The compare result (0 - failure, !0 -
 *                                 success).
 *
 * @retval 0                    The data block was compared successfully,
 *                              compare result is stored into *pfResult.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1abpci_shm_compare_block_16
   (Ix1aSc_t       *pSc,
    unsigned        uAddr,
    const u_int8_t *pbData,
    size_t          nLenData,
    int            *pfResult);





#endif /* __IX1A_BPCI_H */
