/**
 * @file ix1a_hwdefs.h
 *
 * IX1a-HardwareDefinitions - Hardware definitions for all boards.
 *
 * Copyright: 2004 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: ix1a_hwdefs.h,v 1.5.2.1 2005/05/27 16:28:35 thomas Exp $
 * $Project     CAPI for BSD $
 * $Target      ix1a - CAPI manager driver for IX1 active ISDN controllers $
 * @date        11.07.2004
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __IX1A_HWDEFS_H
#define __IX1A_HWDEFS_H





/* === Public declarations =============================================== */





/* --- Common definitions for all board types --- */

/**
 * Identifiers for the boards supported.
 *
 * @important This enumeration corresponds directly to the hardware identifiers
 *            reported by the boards or its firmware.
 *
 * @attention This enumeration is used to index an array of strings in module
 *            ix1a_misc. If this enumeration is modified this names array
 *            must be changed accordingly!
 *
 * @note Currently only Basic ISA, Octo ISA, Multimodem ISA, Basic PCI and
 *       Primary PCI are supported. Especially Micro Channel Architecture (MCA)
 *       and EISA boards are not supported.
 */
typedef enum
{
   IX1A_CARD_TYPE_UNKNOWN       = -1,   /* card type not known */
   IX1A_CARD_TYPE_BASIC_UP0_ISA = 0,    /* Basic ISA, Up0 interface */
   IX1A_CARD_TYPE_BASIC_S0_ISA  = 1,    /* Basic ISA, S0 interface */
   IX1A_CARD_TYPE_BASIC_UP0_MCA = 2,    /* Basic MCA, Up0 interface (not supported) */
   IX1A_CARD_TYPE_BASIC_S0_MCA  = 3,    /* Basic MCA, S0 interface (not supported) */
   IX1A_CARD_TYPE_PRIMARY_EISA  = 4,    /* Primary EISA (not supported) */
   IX1A_CARD_TYPE_PRIMARY_ISA   = 5,    /* Primary ISA (not supported) */
   IX1A_CARD_TYPE_MMOD_ISA      = 6,    /* Multimodem ISA */
   IX1A_CARD_TYPE_OCTO_UP0_ISA  = 7,    /* Octo ISA, Up0 interface */
   IX1A_CARD_TYPE_OCTO_S0_ISA   = 8,    /* Octo ISA, S0 interface */
   IX1A_CARD_TYPE_BASIC_PCI     = 9,    /* Basic PCI */
   IX1A_CARD_TYPE_PRIMARY_PCI   = 10,   /* Primary PCI */
   IX1A_CARD_TYPE_GRANITE_ISA   = 11    /* Digital Modem ISA (not supported) */
} Ix1aCardType_t;



/* --- Definitions for ISA boards --- */

/** The number of reserved ports of an i/o port range. */
#define IX1A_ISA_IO_PORT_RANGE          8

/** Size of the shared memory of ISA boards in bytes. */
#define IX1A_SHM_SIZE_ISA               0x8000



/** @name Registers of ISA boards */
/** @{ */

/** Data register, low byte. */
#define IX1A_REG_DATA_LOW       0x00

/** Data register, high byte. */
#define IX1A_REG_DATA_HIGH      0x01

/** Offset register, low byte. */
#define IX1A_REG_OFFSET_LOW     0x02

/** Offset register, high byte. */
#define IX1A_REG_OFFSET_HIGH    0x03

/** Command register 1 / status register 1. */
#define IX1A_REG_CMD1           0x04

/** Command register 2 / status register 2. */
#define IX1A_REG_CMD2           0x05

/** Hardware type register. */
#define IX1A_REG_HWTYPE         0x06

/** @} */



/** @name Bits of the command / status register 1 */
/** @{ */

/** Bit mask for the IRQ index. */
#define IX1A_CMD1_IRQMASK       0x07

/** Bit for enabling the board. */
#define IX1A_CMD1_ENABLE        0x08

/** 16bit mode switch, if set board is in 16bit mode, else in 8bit mode. */
#define IX1A_CMD1_16BIT         0x20

/** Bit to set reset mode. */
#define IX1A_CMD1_RESET         0x40

/** Bit to set boot mode. */
#define IX1A_CMD1_BOOT          0x80

/** @} */



/** @name Bits of the command / status register 2 */
/** @{ */

/** Bit to set EISA level triggered interrupt mode (not for v1.1 boards). */
#define IX1A_CMD2_EISA_LEVEL_IRQ        0x10

/**
 * Bit to request test interrupt to PC (toggle it) and for detecting a pending
 * interrupt to PC (not for v1.1 boards).
 */
#define IX1A_CMD2_INT2PC                0x20

/** Toggle bit to acknowledge and clear an interrupt to PC. */
#define IX1A_CMD2_CLEAR_INT2PC          0x40

/** Toggle bit to send an interrupt to the board. */
#define IX1A_CMD2_INT2BOARD             0x80

/** @} */



/* --- Definitions for Basic PCI boards --- */

/** PCI vendor id used by ITK. */
#define IX1A_ITK_VENDOR_ID              0x10E8

/** PCI device id for the original ITK Basic PCI board. */
#define IX1A_ITK_BASIC_DEVICE_ID        0x80B1

/** PCI vendor id used for Compaq OEM boards. */
#define IX1A_COMPAQ_VENDOR_ID           0x0E11

/** PCI device id for the Compaq OEM Basic PCI board. */
#define IX1A_COMPAQ_BASIC_DEVICE_ID     0xA0F2

/**
 * Size of the shared memory of Basic PCI boards in bytes (assumed to be the
 * same as for ISA boards).
 */
#define IX1A_SHM_SIZE_BPCI              0x8000



/** @name Registers for Basic PCI boards */
/** @{ */

/**
 * Command and status register 1 (from base of i/o command PCI register).
 * This register has the same meaning as the command and status register 1 for
 * ISA boards. But the 16bit, enable and IRQ assignment bits are not used. The
 * board only supports 16bit mode and always uses INTA.
 */
#define IX1A_REG_PCICMD_CMD1    0x00

/**
 * Offset register to address shared memory (from base of i/o command PCI
 * register.
 */
#define IX1A_REG_PCICMD_OFFSET  0x02

/**
 * PCI header register: Output mailbox register 1, mostly used to send
 * interrupt to the board.
 */
#define IX1A_REG_AMCC_OMB1      0x00

/**
 * PCI header register: Input mailbox register 1, contains status byte after
 * specific operations.
 */
#define IX1A_REG_AMCC_IMB1      0x10

/**
 * PCI header register: 0x01 in the high word is set if an interrupt from the
 * board is pending.
 */
#define IX1A_REG_AMCC_MBEF      0x34

/** PCI header register: Interrupt control status register. */
#define IX1A_REG_AMCC_INTCSR    0x38

/** @} */



/* --- Definitions for Primary PCI boards --- */

/** PCI vendor id used by ITK for the Primary PCI board. */
#define IX1A_GALILEO_VENDOR_ID          0x11AB

/** PCI device id for the Primary PCI sold under the ITK brand. */
#define IX1A_ITK_PRIMARY_DEVICE_ID      0xF002

/** PCI device id for the Primary PCI sold under the Galileo name. */
#define IX1A_GALILEO_PRIMARY_DEVICE_ID  0x0146

/** The size in bytes of the GAL memory resource. */
#define IX1A_PPCI_GAL_MEMORY_SIZE       0x00000D00UL

/** The size in bytes of the shared memory resource. */
#define IX1A_PPCI_SHARED_MEMORY_SIZE    0x00800000UL

/** The size in bytes of the memory mapped register resource. */
#define IX1A_PPCI_MMR_SIZE              0x00800000UL



/** @name Register offsets for the Primary's Galileo chip memory range */
/** @{ */

/** Device bank 3 start address register (dword). */
#define IX1A_PPCI_GAL_DEV3START         0x438

/** Device bank 3 end address register (dword). */
#define IX1A_PPCI_GAL_DEV3END           0x43C

/** PCI configuration register, timeout and retry definitions (dword). */
#define IX1A_PPCI_GAL_PCICONF           0xC04

/** Interrupt register, CPU and PCI bus side (dword). */
#define IX1A_PPCI_GAL_INT               0xC18

/** Interrupt mask register, CPU bus side (dword). */
#define IX1A_PPCI_GAL_CPUINTMASK        0xC1C

/** Interrupt mask register, PCI bus side (dword). */
#define IX1A_PPCI_GAL_PCIINTMASK        0xC24

/** @} */



/** Galileo CPU interrupt mask bit (host to board, 1 << 26). */
#define IX1A_PPCI_GAL_CPUINT_BIT        0x04000000

/** Galileo PCI interrupt mask bit (board to host, 1 << 21). */
#define IX1A_PPCI_GAL_PCIINT_BIT        0x00200000



/** @name Register offsets for the Primary's Memory Mapped Registers */
/** @{ */

/** LED control register (byte). */
#define IX1A_PPCI_MMR_LED               0x00200000

/** Reset register (byte). */
#define IX1A_PPCI_MMR_RESET             0x00400000

/** Status / control register (byte). */
#define IX1A_PPCI_MMR_STATUSCONTROL     0x00600000

/** @} */



/* --- Other definitions --- */

/** The parameter block for the firmware download. */
typedef struct
{
   u_int32_t dwBinSize;         /**< Size of the firmware code in bytes. */
   u_int32_t dwBinBlockSize;    /**<
                                 * The size for a single portion of firmware
                                 * code to send to the board. It may be null
                                 * and will be overwritten during the load
                                 * procedure.
                                 */
   u_int32_t dwBinDestAddr;     /**<
                                 * The destination address for moving the
                                 * firmware code after download.
                                 */
   u_int32_t dwStartAddr;       /**< The kernel start address on the board. */
   u_int32_t dwCheckSum;        /**< Checksum for the firmware code. */
   u_int32_t dwFlags;           /**< Flags, currently not used. */
} __packed Ix1aFirmwareParamBlock_t;



/** @name Firmware status codes during and after firmware startup */
/** @{ */

/** Firmware startup still in progress. */
#define IX1A_FW_STATUS_STARTUP_IN_PROGRESS      0x00

/** Firmware started successfully. */
#define IX1A_FW_STATUS_WORKING                  0x01

/** Firmware crashed, may also be set during operation. */
#define IX1A_FW_STATUS_CRASHED                  0x02

/** Firmware is beeing reloaded. */
#define IX1A_FW_STATUS_RELOAD_REQUEST           0x04

/** Firmware started successfully, but line configuration was rejected. */
#define IX1A_FW_STATUS_WARNING_CONFIGURATION    0x08

/** Firmware startup failure because of invalid line configuration. */
#define IX1A_FW_STATUS_ERROR_CONFIGURATION      0x10

/** Firmware startup failure because of mismatched hardware version. */
#define IX1A_FW_STATUS_ERROR_VERSION            0x20

/** @} */





/* === Prototypes of interface functions ================================= */





#endif /* __IX1A_HWDEFS_H */
