/**
 * @file ix1a_isa.h
 *
 * IX1a-ISA - Routines and specifications for IX1 ISA boards.
 *
 * Copyright: 2004 Thomas Wintergerst. All rights reserved.
 *
 * $Id: ix1a_isa.h,v 1.3.2.1 2005/05/27 16:28:36 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     ix1a - CAPI manager driver for IX1 active ISDN controllers $
 * @date        14.12.2004
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __IX1A_ISA_H
#define __IX1A_ISA_H





/* === Public declarations =============================================== */





/* === Prototypes of interface functions ================================= */





/**
 *  Initialise and fill the resource info structure for an ISA board.
 *
 * This function will initialise the resource info structure for ISA boards
 * and acquire the needed resources.
 *
 * @param dev                   I: The device entry for the board to register.
 * @param pResInfo              I/O: The resource info structure to fill.
 *
 * @retval 0                    All resources could be acquired.
 * @retval Else                 At least one resource could not be acquired.
 *                              In fact no resource is required in this case.
 */
extern int ix1aisa_init_resinfo
   (device_t            dev,
    Ix1aResourceInfo_t *pResInfo);

/**
 * Initialise ISA board for operation.
 *
 * This function is called to initialise an ISA board. The first task is to
 * tell the board its i/o address. Before that the board is completely disabled
 * and does not react on any i/o operation. If the board was initialised
 * before, this step must be left out.
 *
 * Other steps are also necessary to render the board operational. The last one
 * is to turn the board into reset mode. In this state it does nothing and
 * waits for the bootstrap code to be downloaded.
 *
 * @param dev                   I: The device representing the board.
 * @param pSc                   I/O: The softc structure for the board.
 * @param cardType              I: The expected card type. It is used to handle
 *                                 the slight differences between the ISA board
 *                                 types and is checked against the card type
 *                                 delivered by the board itself.
 * @param fFullInit             I:
 *                              - 0
 *                                 Only basic tests are performed. This is used
 *                                 during the probe phase. It will only be
 *                                 checked that there is a board and that it is
 *                                 responding to i/o operations.
 *                              - 1
 *                                 All tests are performed. This includes the
 *                                 on-board shared memory tests in addition to
 *                                 the basic tests. And finally the interrupt
 *                                 routine is enabled.
 *
 * @retval 0                    The tests were successful, board exists and is
 *                              working.
 * @retval Else                 Error occurred, board does not exist or is not
 *                              working.
 */
extern int ix1aisa_init_board
   (device_t        dev,
    Ix1aSc_t       *pSc,
    Ix1aCardType_t  cardType,
    int             fFullInit);

/**
 * Enable an ISA board after disabling it.
 *
 * @note This routine expects the board to have been correctly initialised
 *       before disabling. So no attempt is made to determine the correctness
 *       of the board type.
 *
 * @param pSc                   I/O: The softc structure for the board to
 *                                 enable.
 *
 * @retval 0                    The board was successfully enabled.
 * @retval Else                 Enabling the board failed, the result is an
 *                              errno value.
 */
extern int ix1aisa_enable_board
   (Ix1aSc_t *pSc);

/**
 * Disable an ISA board in general.
 *
 * @param pSc                   I/O: The softc structure for the board to
 *                                 disable.
 *
 * @return Nothing.
 */
extern void ix1aisa_disable_board
   (Ix1aSc_t *pSc);

/**
 * Acknowledge a pending interrupt to a board.
 *
 * This function is called to acknowledge a pending interrupt to the board
 * specified. The acknowledge operation is to toggle the correct bit in the
 * status register 2 shortly from 0 to 1 and back to 0.
 *
 * @param pSc                   I/O: The board to acknowledge the interrupt to.
 *
 * @return Nothing.
 */
extern void ix1aisa_int_ack
   (Ix1aSc_t *pSc);

/**
 * Set the offset pointer within an ISA board's shared memory.
 *
 * @param pSc                   I/O: The softc structure for the board to set
 *                                 the offset register.
 * @param uAddr                 I: The address offset to set.
 *
 * @retval 0                    The offset register was set successfully.
 * @retval Else                 The offset register could not be set, board
 *                              failure.
 */
extern int ix1aisa_shm_set_offset
   (Ix1aSc_t *pSc,
    unsigned  uAddr);

/**
 * Read a byte from a shared memory address of an ISA board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to read from.
 * @param puValue               O: The byte read from the board as an unsigned
 *                                 integer, only valid if the result is 0.
 *
 * @retval 0                    The value was read successfully and stored into
 *                              *puValue.
 * @retval Else                 An errno value on failure.
 */
extern int ix1aisa_shm_read_byte
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    unsigned *puValue);

/**
 * Read a 16bit word from a shared memory address of an ISA board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to read from.
 * @param puValue               O: The word read from the board, only valid if
 *                                 the result is 0.
 *
 * @retval 0                    The value was read successfully and stored into
 *                              *puValue.
 * @retval Else                 An errno value on failure.
 */
extern int ix1aisa_shm_read_word
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    unsigned *puValue);

/**
 * Read a block of data from a shared memory address of an ISA board assuming
 * 16bit mode.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to read from.
 * @param pbData                I: The address for returing the data read.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 read.
 *
 * @retval 0                    The data block was read successfully.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1aisa_shm_read_block_16
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    u_int8_t *pbData,
    size_t    nLenData);

/**
 * Write a 16bit word to a shared memory address of an ISA board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param uValue                I: The word to write.
 *
 * @retval 0                    The value was successfully written.
 * @retval Else                 An errno value on failure.
 */
extern int ix1aisa_shm_write_word
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    unsigned  uValue);

/**
 * Write a block of data to a shared memory address of an ISA board assuming
 * 8bit mode.
 *
 * @pre This function may only be called for Basic and Octo boards, Multimodems
 *      do not support 8bit mode.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param pbData                I: The address of the data block to write.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 write.
 *
 * @retval 0                    The data block was written successfully.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1aisa_shm_write_block_8
   (Ix1aSc_t       *pSc,
    unsigned        uAddr,
    const u_int8_t *pbData,
    size_t          nLenData);

/**
 * Write a block of data to a shared memory address of an ISA board assuming
 * 16bit mode.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param pbData                I: The address of the data block to write.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 write.
 *
 * @retval 0                    The data block was written successfully.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1aisa_shm_write_block_16
   (Ix1aSc_t       *pSc,
    unsigned        uAddr,
    const u_int8_t *pbData,
    size_t          nLenData);

/**
 * Compare a block of data to a shared memory address of an ISA board assuming
 * 16bit mode.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param pbData                I: The address of the data block to write.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 write.
 * @param pfResult              O: The compare result (0 - failure, !0 -
 *                                 success).
 *
 * @retval 0                    The data block was compared successfully,
 *                              compare result is stored into *pfResult.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1aisa_shm_compare_block_16
   (Ix1aSc_t       *pSc,
    unsigned        uAddr,
    const u_int8_t *pbData,
    size_t          nLenData,
    int            *pfResult);





#endif /* __IX1A_ISA_H */
