/**
 * @file ix1a_misc.h
 *
 * Ix1a-Misc - Miscellaneous helper functions for the ix1a device driver.
 *
 * Copyright: 2004 Thomas Wintergerst. All rights reserved.
 *
 * $Id: ix1a_misc.h,v 1.4.2.1 2005/05/27 16:28:36 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     ix1a - CAPI manager driver for IX1 active ISDN controllers $
 * @date        17.08.2004
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __IX1A_MISC_H
#define __IX1A_MISC_H

#include <c4b/driver/ix1a/ix1a_global.h>
#include <c4b/driver/ix1a/ix1a_hwdefs.h>





/* === Public declarations =============================================== */





/* --- some strings for naming synchronization objects --- */

/** The name of the access mutex for a controller. */
#define IX1AMISC_ACCESS_MUTEX_MSG       "ix1a access mutex"

/** The name of the condition object for signalling of events. */
#define IX1AMISC_NOTIFY_CONDITION_MSG   "ix1a notify condition"



/* --- some strings for the tsleep call --- */

/** The string to display while waiting for exclusive controller access. */
#define IX1AMISC_WAIT_FOR_CTLR_ACCESS_MSG       "wait for ctlr access"

/** String for msleep() while waiting for completed firmware startup. */
#define IX1AMISC_FW_READY_WAIT_STRING           "Firmware startup waiting"



/* --- some helper macros --- */

#define ARRAY_COUNT( _a )       (sizeof (_a) / sizeof ((_a) [0]))





/* === Prototypes of interface functions ================================= */





/* --- serializing access to a controller --- */

/**
 * Get access to a controller for an operation.
 *
 * @note This function is called by all CAPI manager entry functions. The
 *       interrupt routine must not call it, but must directly obtain the board
 *       access mutex.
 *
 * @param pSc                   I/O: The softc structure for the board.
 *
 * @retval 0                    Unable to obtain board access, timeout.
 * @retval 1                    Exclusive board access gained.
 */
extern int ix1amisc_get_ctlr_access
   (Ix1aSc_t *pSc);

/**
 * Release access to a controller after an operation.
 */
extern void ix1amisc_release_ctlr_access
   (Ix1aSc_t *pSc);



/* --- Routines common to all board types --- */

/**
 * Release all resources allocated within a resource info structure.
 *
 * This function will release all resources still allocated within the
 * structure given as an argument.
 *
 * @param dev                   I: The device entry for the board to register.
 * @param pResInfo              I/O: The resource info with the resources to
 *                                 release.
 */
extern int ix1amisc_release_resinfo
   (device_t            dev,
    Ix1aResourceInfo_t *pResInfo);

/**
 * Perform a download operation for a board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param nNumDataBlocks        I: The number of data blocks for the download
 *                                 operation. If not only a board reset is
 *                                 requested, there must be three data blocks
 *                                 at paDataBlocks.
 * @param paDataBlocks          I: The array of data blocks for the download
 *                                 operation. For a pure reset operation no
 *                                 data block is specified. Else there must be
 *                                 three data blocks: Primary bootstrap,
 *                                 firmware and line configuration.
 *
 * @retval 0                    The reset or download operation was successful.
 * @retval Else                 The reset or download operation failed, result
 *                              is an errno value.
 */
extern int ix1amisc_download_board
   (Ix1aSc_t                  *pSc,
    size_t                     nNumDataBlocks,
    const CAPICtlrDataBlock_t *paDataBlocks);

/**
 * Check a data block for valid board firmware.
 *
 * A valid firmware data block starts with two Ctrl-Z terminated strings, the
 * first must start with the text "Loadfile". This is followed by a parameter
 * block of six dwords. The real length is prepended to the parameter block as
 * a single byte right after the 2nd Ctrl-Z. Behind the parameter block the
 * real firmware follows. Normally it continues to the end of the data block.
 * But the first dword of the parameter block determines the number of bytes to
 * send to the board. There may be some bytes left behind the firmware.
 *
 * @param pSc                   I/O: The softc structure for the board context.
 * @param pbDataBlock           I: The firmware code block to check.
 * @param nLenDataBlock         I: The length of the firmware data block at
 *                                 pbDataBlock.
 * @param pParamBlock           O: On successful return the content of the
 *                                 parameter block found is stored here.
 * @param ppbFirmwareCode       O: On successful return the address of the real
 *                                 firmware code is returned here.
 * @param pnLenFirmwareCode     O: On successful return the real length of the
 *                                 firmware to send to the board in bytes is
 *                                 returned here.
 *
 * @retval 0                    The firmware code data block is valid,
 *                              *pParamBlock, *ppFirmwareCode and
 *                              *pnLenFirmwareCode are filled with the values
 *                              determined.
 * @retval Else                 The firmware code data block is invalid, the
 *                              result is an appropriate errno value.
 */
extern int ix1amisc_check_firmware_code
   (Ix1aSc_t                  *pSc,
    const u_int8_t            *pbDataBlock,
    size_t                     nLenDataBlock,
    Ix1aFirmwareParamBlock_t  *pParamBlock,
    const u_int8_t           **ppbFirmwareCode,
    size_t                    *pnLenFirmwareCode);



/* --- some helper functions --- */

/**
 * Translate a card type into a string.
 */
extern const char *ix1amisc_get_card_type_name
   (Ix1aCardType_t cardType);





#endif /* __IX1A_MISC_H */
