/**
 * @file ix1a_ppci.h
 *
 * IX1a-PrimaryPCI - Routines and specifications for IX1 Primary PCI boards.
 *
 * Copyright: 2004 Thomas Wintergerst. All rights reserved.
 *
 * $Id: ix1a_ppci.h,v 1.3.2.1 2005/05/27 16:28:37 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     ix1a - CAPI manager driver for IX1 active ISDN controllers $
 * @date        15.12.2004
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __IX1A_PPCI_H
#define __IX1A_PPCI_H





/* === Public declarations =============================================== */





/* === Prototypes of interface functions ================================= */





/**
 *  Initialise and fill the resource info structure for a Primary PCI board.
 *
 * This function will initialise the resource info structure for Primary PCI
 * boards and acquire the needed resources.
 *
 * @param dev                   I: The device entry for the board to register.
 * @param pResInfo              I/O: The resource info structure to fill.
 *
 * @retval 0                    All resources could be acquired.
 * @retval Else                 At least one resource could not be acquired.
 *                              In fact no resource is required in this case.
 */
extern int ix1appci_init_resinfo
   (device_t            dev,
    Ix1aResourceInfo_t *pResInfo);

/**
 * Initialise Primary PCI board for operation.
 *
 * @param dev                   I: The device representing the board.
 * @param pSc                   I/O: The softc structure for the board.
 *
 * @retval 0                    The tests were successful, board exists and is
 *                              working.
 * @retval Else                 Error occurred, board does not exist or is not
 *                              working.
 */
extern int ix1appci_init_board
   (device_t  dev,
    Ix1aSc_t *pSc);

/**
 * Disable a Primary PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board to
 *                                 disable.
 *
 * @return Nothing.
 */
extern void ix1appci_disable_board
   (Ix1aSc_t *pSc);

/**
 * Perform a download operation for a Primary PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param nNumDataBlocks        I: The number of data blocks for the download
 *                                 operation. If not only a board reset is
 *                                 requested, there must be three data blocks
 *                                 at paDataBlocks.
 * @param paDataBlocks          I: The array of data blocks for the download
 *                                 operation. For a pure reset operation no
 *                                 data block is specified. Else there must be
 *                                 three data blocks: Primary bootstrap,
 *                                 firmware and line configuration.
 *
 * @retval 0                    The reset or download operation was successful.
 * @retval Else                 The reset or download operation failed, result
 *                              is an errno value.
 */
extern int ix1appci_download_board
   (Ix1aSc_t                  *pSc,
    size_t                     nNumDataBlocks,
    const CAPICtlrDataBlock_t *paDataBlocks);

/**
 * Read a byte from a shared memory address of a Primary PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to read from.
 * @param puValue               O: The byte read from the board as an unsigned
 *                                 integer, only valid if the result is 0.
 *
 * @retval 0                    The value was read successfully and stored into
 *                              *puValue.
 * @retval Else                 An errno value on failure.
 */
extern int ix1appci_shm_read_byte
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    unsigned *puValue);

/**
 * Read a 16bit word from a shared memory address of a Primary PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to read from.
 * @param puValue               O: The word read from the board, only valid if
 *                                 the result is 0.
 *
 * @retval 0                    The value was read successfully and stored into
 *                              *puValue.
 * @retval Else                 An errno value on failure.
 */
extern int ix1appci_shm_read_word
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    unsigned *puValue);

/**
 * Read a block of data from a shared memory address of a Primary PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to read from.
 * @param pbData                I: The address for returing the data read.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 read.
 *
 * @retval 0                    The data block was read successfully.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1appci_shm_read_block
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    u_int8_t *pbData,
    size_t    nLenData);

/**
 * Write a 16bit word to a shared memory address of a Primary PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param uValue                I: The word to write.
 *
 * @retval 0                    The value was successfully written.
 * @retval Else                 An errno value on failure.
 */
extern int ix1appci_shm_write_word
   (Ix1aSc_t *pSc,
    unsigned  uAddr,
    unsigned  uValue);

/**
 * Write a block of data to a shared memory address of a Primary PCI board.
 *
 * @param pSc                   I/O: The softc structure for the board.
 * @param uAddr                 I: The shared memory address to write to.
 * @param pbData                I: The address of the data block to write.
 * @param nLenData              I: The length in bytes of the data block to
 *                                 write.
 *
 * @retval 0                    The data block was written successfully.
 * @retval Else                 Failure, result is an errno value.
 */
extern int ix1appci_shm_write_block
   (Ix1aSc_t       *pSc,
    unsigned        uAddr,
    const u_int8_t *pbData,
    size_t          nLenData);





#endif /* __IX1A_PPCI_H */
