/**
 * @file capi20.h
 *
 * CAPI20 - (Nearly) Platform independent definitions for CAPI (messages).
 *
 * Copyright: 2002-2005 Thomas Wintergerst. All rights reserved.
 *
 * $Id: capi20.h,v 1.15.2.1 2005/05/27 16:29:05 thomas Exp $
 * $Project:    CAPI $
 * $Target:     capi-header $
 * @date    03.08.2002
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __CAPI20_H
#define __CAPI20_H

#ifndef __CAPIDEF_H
#  include <capidef.h>
#endif
#ifndef __CAPI_RESULT_H
#  include <capi_result.h>
#endif
#ifndef __CAPI_SUPPLSVC_H
#  include <capi_supplsvc.h>
#endif
#ifndef __CAPI_LI_H
#  include <capi_li.h>
#endif
#ifndef __CAPI_ECHOC_H
#  include <capi_echoc.h>
#endif

#include <capi_pack_on.h>





/* === public declarations =============================================== */





/* --- definition of the message commands --- */

/* usage of message subcommand + command   */
/* example:                                */
/*   if( cmd == (DATA_B3 | REQ | CAPI20) ) */
/*      ...                                */

/* capi command version */
#define  CAPI20                         0x8000

/* sub commands */
#define  C_REQ                          0x0000
#define  C_CONF                         0x0100
#define  C_IND                          0x0200
#define  C_RESP                         0x0300

/* commands */
#define  C_ALERT                        0x0001
#define  C_CONNECT                      0x0002
#define  C_CONNECT_ACTIVE               0x0003
#define  C_DISCONNECT                   0x0004
#define  C_LISTEN                       0x0005
#define  C_INFO_20                      0x0008
#define  C_SELECT_B_PROTOCOL            0x0041
#define  C_FACILITY                     0x0080
#define  C_CONNECT_B3                   0x0082
#define  C_CONNECT_B3_ACTIVE            0x0083
#define  C_DISCONNECT_B3                0x0084
#define  C_DATA_B3                      0x0086
#define  C_RESET_B3_20                  0x0087
#define  C_CONNECT_B3_T90_ACTIVE        0x0088
#define  C_MANUFACTURER                 0x00FF

/* helpers for extracting parts of a CAPI message command */
#define CAPI_CMDMASK_COMMAND    0x00FF
#define CAPI_CMDMASK_SUBCMD     0x0F00

/* helpers for creating CAPI message commands */
#define CAPI_REQUEST( cmd )     ((cmd) | C_REQ | CAPI20)
#define CAPI_CONFIRM( cmd )     ((cmd) | C_CONF | CAPI20)
#define CAPI_INDICAT( cmd )     ((cmd) | C_IND | CAPI20)
#define CAPI_RESPONSE( cmd )    ((cmd) | C_RESP | CAPI20)



/* --- definition of the CAPI messages --- */

/* message fixed part (header) */
typedef struct CAPIMsgHead
{
   cWORD  wLen;
   cWORD  wApp;
   cWORD  wCmd;
   cWORD  wNum;
   cDWORD dwCid;       /* connection identifier (NCCI,PLCI,CTLR) */
} CAPIMsgHead_t;

/* CAPI header access macros */
#define CAPI_GET_LEN( pMsg )            C_GET_WORD( (pMsg)->head.wLen )
#define CAPI_GET_APPL( pMsg )           C_GET_WORD( (pMsg)->head.wApp )
#define CAPI_GET_CMD( pMsg )            C_GET_WORD( (pMsg)->head.wCmd )
#define CAPI_GET_MSGNUM( pMsg )         C_GET_WORD( (pMsg)->head.wNum )
#define CAPI_GET_CID( pMsg )            C_GET_DWORD( (pMsg)->head.dwCid )

/* helpers for accessing each member of a connection identifier */
#define CAPI_CIDMASK_CTLR       0x0000007FUL
#define CAPI_CIDMASK_CTLR_TYPE  0x00000080UL
#define CAPI_CIDMASK_PLCI       0x0000FF00UL
#define CAPI_CIDMASK_NCCI       0xFFFF0000UL
#define CAPI_CIDSHIFT_PLCI      8
#define CAPI_CIDSHIFT_NCCI      16

/** Extract the controller from a CID. */
#define CAPI_GET_CTLR_FROM_CID( dwCid ) \
   ((dwCid) & CAPI_CIDMASK_CTLR)

/** Extract the PLCI from a CID. */
#define CAPI_GET_PLCI_FROM_CID( dwCid ) \
   (((dwCid) & CAPI_CIDMASK_PLCI) >> CAPI_CIDSHIFT_PLCI)

/** Extract the NCCI from a CID. */
#define CAPI_GET_NCCI_FROM_CID( dwCid ) \
   (((dwCid) & CAPI_CIDMASK_NCCI) >> CAPI_CIDSHIFT_NCCI)



/* messages info parts */

/* Alert-Request */
/* tyepdef struct CAPIAlertReq          */
/* {                                    */
/*    CAPIStruct  AdditionalInfo;       */
/* } CAPIAlertReq_t;                    */

/* Alert-Confirm */
typedef struct CAPIAlertConf
{
   cWORD        wInfo;
} CAPIAlertConf_t;

/* Connect-Request */
typedef struct CAPIConnectReq
{
   cWORD        wCip;
/* CAPIStruct   CalledPartyNumber;      */
/* CAPIStruct   CallingPartyNumber;     */
/* CAPIStruct   CalledPartySubAdr;      */
/* CAPIStruct   CallingPartySubAdr;     */
/* CAPIStruct   BProtocol;              */
/* CAPIStruct   BC;                     */
/* CAPIStruct   LLC;                    */
/* CAPIStruct   HLC;                    */
/* CAPIStruct   AdditionalInfo;         */
} CAPIConnectReq_t;

/* Connect-Confirm */
typedef struct CAPIConnectConf
{
   cWORD        wInfo;
} CAPIConnectConf_t;

/* Connect-Indication */
typedef struct CAPIConnectInd
{
   cWORD        wCip;
/* CAPIStruct   CalledPartyNumber;      */
/* CAPIStruct   CallingPartyNumber;     */
/* CAPIStruct   CalledPartySubAdr;      */
/* CAPIStruct   CallingPartySubAdr;     */
/* CAPIStruct   BC;                     */
/* CAPIStruct   LLC;                    */
/* CAPIStruct   HLC;                    */
/* CAPIStruct   AdditionalInfo;         */
/* CAPIStruct   SecondCallingPartyNumber*/ /* may not be present */
} CAPIConnectInd_t;

/* Connect-Response */
typedef struct CAPIConnectResp
{
   cWORD        wReject;
/* CAPIStruct   BProtocol;              */
/* CAPIStruct   ConnectedPartyNumber;   */
/* CAPIStruct   ConnectedPartySubAdr;   */
/* CAPIStruct   LLC;                    */
/* CAPIStruct   AdditionalInfo;         */
} CAPIConnectResp_t;

/* Connect-Active-Indication */
/* typedef struct CAPIConnectActiveInd  */
/* {                                    */
/*    CAPIStruct  ConnectedPartyNumber; */
/*    CAPIStruct  ConnectedPartySubAdr; */
/*    CAPIStruct  LLC;                  */
/* } CAPIConnectActiveInd_t; */

/* Connect-Active-Response */
/* typedef struct CAPIConnectActiveResp */
/* {                                    */
/* } CAPIConnectActiveResp_t; */

/* Connect-B3-Active-Indication */
/* typedef struct CAPIConnectB3ActiveInd*/
/* {                                    */
/*    CAPIStruct  NCPI;                 */
/* } CAPIConnectB3ActiveInd_t; */

/* Connect-B3-Active-Response */
/* typedef struct CAPIConnectB3ActiveResp */
/* {                                    */
/* } CAPIConnectB3ActiveResp_t; */

/* Connect-B3-Request */
/* typedef struct CAPIConnectB3Req      */
/* {                                    */
/*    CAPIStruct  NCPI;                 */
/* } CAPIConnectB3Req_t; */

/* Connect-B3-Confirm */
typedef struct CAPIConnectB3Conf
{
   cWORD        wInfo;
} CAPIConnectB3Conf_t;

/* Connect-B3-Indication */
/* typedef struct CAPIConnectB3Ind      */
/* {                                    */
/*    CAPIStruct  NCPI;                 */
/* } CAPIConnectB3Ind_t; */

/* Connect-B3-Response */
typedef struct CAPIConnectB3Resp
{
   cWORD        wReject;
/* CAPIStruct   NCPI;                   */
} CAPIConnectB3Resp_t;

/* Connect-B3-T90-Active-Indication */
/* typedef struct CAPIConnectB3T90ActiveInd */
/* {                                    */
/*    CAPIStruct  NCPI;                 */
/* } CAPIConnectB3T90ActiveInd_t; */

/* Connect-B3-T90-Active-Response */
/* typedef struct CAPIConnectB3T90ActiveResp */
/* {                                    */
/* } CAPIConnectB3T90ActiveResp_t; */

/**
 * Structure of a Data-B3-Request for 32bit applications.
 *
 * @important The layout of this structure must be identical to the one for
 *            64bit applications and the combined 32 and 64 bit structure.
 */
typedef struct CAPIDataB332Req
{
   cDWORD dwData;               /**< 32bit address of the data block. */
   cWORD  wLen;                 /**< Length of the data block in bytes. */
   cWORD  wHandle;              /**< Appl. defined current data block no. */
   cWORD  wFlags;               /**<
                                 * Bit field, values from CAPI_DATA_B3_FLAG_*.
                                 */
} CAPIDataB332Req_t;

/**
 * Structure of a Data-B3-Request for 64bit applications.
 *
 * @important The layout of this structure must be identical to the one for
 *            32bit applications and the combined 32 and 64 bit structure.
 */
typedef struct CAPIDataB364Req
{
   cDWORD dwDataUnused;         /**<
                                 * 32bit address of the data block that is not
                                 * used for 64bit applications.
                                 */
   cWORD  wLen;                 /**< Length of the data block in bytes. */
   cWORD  wHandle;              /**< Appl. defined current data block no. */
   cWORD  wFlags;               /**<
                                 * Bit field, values from CAPI_DATA_B3_FLAG_*.
                                 */
   cQWORD qwData64;             /**< 64bit address of data block. */
} CAPIDataB364Req_t;

/**
 * Structure of a Data-B3-Request for 32 and 64 bit applications.
 *
 * @important The layout of this structure must be identical to the ones for 32
 *            and 64 bit applications.
 */
typedef struct CAPIDataB3Req
{
   cDWORD dwData;               /**<
                                 * 32bit address of the data block. On 64bit
                                 * machines this field must not be used, it must
                                 * be set to 0. The real address must be placed
                                 * into qwData64.
                                 */
   cWORD  wLen;                 /**< Length of the data block in bytes. */
   cWORD  wHandle;              /**< Appl. defined current data block no. */
   cWORD  wFlags;               /**<
                                 * Bit field, values from CAPI_DATA_B3_FLAG_*.
                                 */
   cQWORD qwData64;             /**<
                                 * 64bit address of data block, if dwData is
                                 * not used, i.e. 0.
                                 */
} CAPIDataB3Req_t;

/** Structure of a Data-B3-Confirm. */
typedef struct CAPIDataB3Conf
{
   cWORD wHandle;               /**<
                                 * The handle of the confirmed data block.
                                 * This is the same value as wHandle in the
                                 * corresponding Data-B3-Request.
                                 */
   cWORD wInfo;                 /**< The CAPI result value for the request. */
} CAPIDataB3Conf_t;

/**
 * Structure of a Data-B3-Indication for 32bit applications.
 *
 * @important The layout of this structure must be identical to the one for
 *            64bit applications and the combined 32 and 64 bit structure.
 */
typedef struct CAPIDataB332Ind
{
   cDWORD dwData;               /**< 32bit address of the data block. */
   cWORD  wLen;                 /**< Length of the data block in bytes. */
   cWORD  wHandle;              /**< Ctlr. defined current data block no. */
   cWORD  wFlags;               /**<
                                 * Bit field, values from CAPI_DATA_B3_FLAG_*.
                                 */
} CAPIDataB332Ind_t;

/**
 * Structure of a Data-B3-Indication for 64bit applications.
 *
 * @important The layout of this structure must be identical to the one for
 *            32bit applications and the combined 32 and 64 bit structure.
 */
typedef struct CAPIDataB364Ind
{
   cDWORD dwDataUnused;         /**<
                                 * 32bit address of the data block that is not
                                 * used for 64bit applications.
                                 */
   cWORD  wLen;                 /**< Length of the data block in bytes. */
   cWORD  wHandle;              /**< Ctlr. defined current data block no. */
   cWORD  wFlags;               /**<
                                 * Bit field, values from CAPI_DATA_B3_FLAG_*.
                                 */
   cQWORD qwData64;             /**< 64bit address of data block. */
} CAPIDataB364Ind_t;

/**
 * Structure of a Data-B3-Indication for 32 and 64 bit applications.
 *
 * @important The layout of this structure must be identical to the ones for 32
 *            and 64 bit applications.
 */
typedef struct CAPIDataB3Ind
{
   cDWORD dwData;               /**<
                                 * 32bit address of the data block. On 64bit
                                 * machines this field is not used and is set to
                                 * 0. The real address is placed into qwData64.
                                 */
   cWORD  wLen;                 /**< Length of the data block in bytes. */
   cWORD  wHandle;              /**< Ctlr. defined current data block no. */
   cWORD  wFlags;               /**<
                                 * Bit field, values from CAPI_DATA_B3_FLAG_*.
                                 */
   cQWORD qwData64;             /**<
                                 * 64bit address of data block, if dwData is
                                 * not used, i.e. 0.
                                 */
} CAPIDataB3Ind_t;

/** Structure of a Data-B3-Response. */
typedef struct CAPIDataB3Resp
{
   cWORD wHandle;               /**<
                                 * The handle of the acknowledged data block.
                                 * This must be the same value as wHandle in the
                                 * corresponding Data-B3-Indication.
                                 */
} CAPIDataB3Resp_t;

/* Disconnect-B3-Request */
/* typedef struct CAPIDisconnectB3Req   */
/* {                                    */
/*    CAPIStruct  NCPI;                 */
/* } CAPIDisconnectB3Req_t; */

/* Disconnect-B3-Confirm */
typedef struct CAPIDisconnectB3Conf
{
   cWORD        wInfo;
} CAPIDisconnectB3Conf_t;

/* Disconnect-B3-Indication */
typedef struct CAPIDisconnectB3Ind
{
   cWORD        wReason;
/* CAPIStruct   NCPI;                   */
} CAPIDisconnectB3Ind_t;

/* Disconnect-B3-Response */
/* typedef struct CAPIDisconnectB3Resp  */
/* {                                    */
/* } CAPIDisconnectB3Resp_t; */

/* Disconnect-Request */
/* typedef struct CAPIDisconnectReq     */
/* {                                    */
/*    CAPIStruct  AdditionalInfo;       */
/* } CAPIDisconnectReq_t; */

/* Disconnect-Confirm */
typedef struct CAPIDisconnectConf
{
   cWORD        wInfo;
} CAPIDisconnectConf_t;

/* Disconnect-Indication */
typedef struct CAPIDisconnectInd
{
   cWORD        wReason;
} CAPIDisconnectInd_t;

/* Disconnect-Response */
/* typedef struct CAPIDisconnectResp    */
/* {                                    */
/* } CAPIDisconnectResp_t; */

/* Facility-Request */
typedef struct CAPIFacilityReq
{
   cWORD        wSelector;
/* CAPIStruct   FacilityReqParam;       */
} CAPIFacilityReq_t;

/* Facility-Confirm */
typedef struct CAPIFacilityConf
{
   cWORD        wInfo;
   cWORD        wSelector;
/* CAPIStruct   FacilityConfParam;      */
} CAPIFacilityConf_t;

/* Facility-Indication */
typedef struct CAPIFacilityInd
{
   cWORD        wSelector;
/* CAPIStruct   FacilityIndParam;       */
} CAPIFacilityInd_t;

/* Facility-Response */
typedef struct CAPIFacilityResp
{
   cWORD        wSelector;
/* CAPIStruct   FacilityRespParam;      */
} CAPIFacilityResp_t;

/* Info-Request */
/* typedef struct CAPIInfoReq           */
/* {                                    */
/*    CAPIStruct  CalledPartyNumber;    */
/*    CAPIStruct  AdditionalInfo;       */
/* } CAPIInfoReq_t;    */

/* Info-Confirm */
typedef struct CAPIInfoConf
{
   cWORD        wInfo;
} CAPIInfoConf_t;

/* Info-Indication */
typedef struct CAPIInfoInd
{
   cWORD        wInfoNum;
/* CAPIStruct   InfoElement;            */
} CAPIInfoInd_t;

/* Info-Response */
/* typedef struct CAPIInfoResp          */
/* {                                    */
/* } CAPIInfoResp_t;   */

/* Listen-Request */
typedef struct CAPIListenReq
{
   cDWORD       dwInfoMask;             /* bit field of requested information,
                                         * values from CAPI_INFO_MASK_*
                                         */
   cDWORD       dwCipMask;              /* bit field of wanted services for
                                         * incoming calls, values from
                                         * CAPI_CIP_MASK_*
                                         */
   cDWORD       dwCipMask2;             /* bit field of wanted services for
                                         * incoming calls (extent of
                                         * dwCipMask), values from
                                         * CAPI_CIP_MASK2_*
                                         */
/* CAPIStruct   CallingPartyNumber;     */
/* CAPIStruct   CallingPartySubAdr;     */
} CAPIListenReq_t;

/* Listen-Confirm */
typedef struct CAPIListenConf
{
   cWORD        wInfo;
} CAPIListenConf_t;

/* Manufacturer-Request */
typedef struct CAPIManufacturerReq
{
   cDWORD       dwManuID;
   /* manufacturer-specific data follows */
} CAPIManufacturerReq_t;

/* Manufacturer-Confirm */
typedef struct CAPIManufacturerConf
{
   cDWORD       dwManuID;
   /* manufacturer-specific data follows */
} CAPIManufacturerConf_t;

/* Manufacturer-Indication */
typedef struct CAPIManufacturerInd
{
   cDWORD       dwManuID;
   /* manufacturer-specific data following */
} CAPIManufacturerInd_t;

/* Manufacturer-Response */
typedef struct CAPIManufacturerResp
{
   cDWORD       dwManuID;
   /* manufacturer-specific data following */
} CAPIManufacturerResp_t;

/* Reset-B3-Request */
/* typedef struct CAPIResetB3Req        */
/* {                                    */
/*    CAPIStruct  NCPI;                 */
/* } CAPIResetB3Req_t; */

/* Reset-B3-Confirm */
typedef struct CAPIResetB3Conf
{
   cWORD        wInfo;
} CAPIResetB3Conf_t;

/* Reset-B3-Indication */
/* typedef struct CAPIResetB3Ind        */
/* {                                    */
/*    CAPIStruct  NCPI;                 */
/* } CAPIResetB3Ind_t; */

/* Reset-B3-Response */
/* typedef struct CAPIResetB3Resp       */
/* {                                    */
/* } CAPIResetB3Resp_t;*/

/* Select-B-Protocol-Request */
/* typedef struct CAPISelectBProtocolReq*/
/* {                                    */
/*    CAPIStruct  BProtocol;            */
/* } CAPISelectBProtocolReq_t; */

/* Select-B-Protocol-Confirm */
typedef struct CAPISelectBProtocolConf
{
   cWORD        wInfo;
} CAPISelectBProtocolConf_t;

/* Some CAPI message without pre-defined layout */
/* Note: When the message header is added, the size is equal to 1024 bytes */
typedef union CAPIAnyMsg
{
   cBYTE     b [1012];
   cWORD     w [506];
   cDWORD    d [253];
   cQWORD    q [126];
} CAPIAnyMsg_t;



/* Main definition of a CAPI message as a summation of all structures defined
 * above
 */
typedef struct CAPIMsg
{
   struct CAPIMsgHead head;
   union
   {
      union  CAPIAnyMsg                     any;
/*    struct CAPIAlertReq                   alert_req;                      */
      struct CAPIAlertConf                  alert_conf;
      struct CAPIConnectReq                 connect_req;
      struct CAPIConnectConf                connect_conf;
      struct CAPIConnectInd                 connect_ind;
      struct CAPIConnectResp                connect_resp;
/*    struct CAPIConnectActiveInd           connect_active_ind;             */
/*    struct CAPIConnectActiveResp          connect_active_resp;            */
/*    struct CAPIConnectB3ActiveInd         connect_b3_active_ind;          */
/*    struct CAPIConnectB3ActiveResp        connect_b3_active_resp;         */
/*    struct CAPIConnectB3Req               connect_b3_req;                 */
      struct CAPIConnectB3Conf              connect_b3_conf;
/*    struct CAPIConnectB3Ind               connect_b3_ind;                 */
      struct CAPIConnectB3Resp              connect_b3_resp;
/*    struct CAPIConnectB3T90ActiveInd      connect_b3_t90_active_ind;      */
/*    struct CAPIConnectB3T90ActiveResp     connect_b3_t90_active_resp;     */
      struct CAPIDataB332Req                data_b3_32_req;
      struct CAPIDataB364Req                data_b3_64_req;
      struct CAPIDataB3Req                  data_b3_req;
      struct CAPIDataB3Conf                 data_b3_conf;
      struct CAPIDataB332Ind                data_b3_32_ind;
      struct CAPIDataB364Ind                data_b3_64_ind;
      struct CAPIDataB3Ind                  data_b3_ind;
      struct CAPIDataB3Resp                 data_b3_resp;
/*    struct CAPIDisconnectB3Req            disconnect_b3_req;              */
      struct CAPIDisconnectB3Conf           disconnect_b3_conf;
      struct CAPIDisconnectB3Ind            disconnect_b3_ind;
/*    struct CAPIDisconnectB3Resp           disconnect_b3_resp;             */
/*    struct CAPIDisconnectReq              disconnect_req;                 */
      struct CAPIDisconnectConf             disconnect_conf;
      struct CAPIDisconnectInd              disconnect_ind;
/*    struct CAPIDisconnectResp             disconnect_resp;                */
      struct CAPIFacilityReq                facility_req;
      struct CAPIFacilityConf               facility_conf;
      struct CAPIFacilityInd                facility_ind;
      struct CAPIFacilityResp               facility_resp;
/*    struct CAPIInfoReq                    info_req;                       */
      struct CAPIInfoConf                   info_conf;
      struct CAPIInfoInd                    info_ind;
/*    struct CAPIInfoResp                   info_resp;                      */
      struct CAPIListenReq                  listen_req;
      struct CAPIListenConf                 listen_conf;
      struct CAPIManufacturerReq            manufacturer_req;
      struct CAPIManufacturerConf           manufacturer_conf;
      struct CAPIManufacturerInd            manufacturer_ind;
      struct CAPIManufacturerResp           manufacturer_resp;
/*    struct CAPIResetB3Req                 reset_b3_req;                   */
      struct CAPIResetB3Conf                reset_b3_conf;
/*    struct CAPIResetB3Ind                 reset_b3_ind;                   */
/*    struct CAPIResetB3Resp                reset_b3_resp;                  */
/*    struct CAPISelectBProtocolReq         select_b_protocol_req;          */
      struct CAPISelectBProtocolConf        select_b_protocol_conf;
   } info;
} CAPIMsg_t;



/* --- B-channel protocols --- */

/* B1 protocols */
#define CAPI_B1_HDLC_64                 0     /* default */
#define CAPI_B1_TRANSPARENT_64          1
#define CAPI_B1_V110_ASYNCH             2
#define CAPI_B1_V110_SYNCH_HDLC         3
#define CAPI_B1_T30_MODEM               4
#define CAPI_B1_HDLC_64_INVERTED        5
#define CAPI_B1_TRANSPARENT_56          6
#define CAPI_B1_MODEM_FULL              7
#define CAPI_B1_MODEM_ASYNCH            8
#define CAPI_B1_MODEM_SYNCH_HDLC        9

/* B2 protocols */
#define CAPI_B2_ISO7776_X75_SLP         0     /* default */
#define CAPI_B2_TRANSPARENT             1
#define CAPI_B2_SDLC                    2
#define CAPI_B2_LAPD_X25                3
#define CAPI_B2_T30_FAX_G3              4
#define CAPI_B2_PPP                     5
#define CAPI_B2_TRANSPARENT_NO_B1_ERR   6
#define CAPI_B2_MODEM_FULL              7
#define CAPI_B2_ISO7776_X75_SLP_COMP    8
#define CAPI_B2_V120_ASYNCH             9
#define CAPI_B2_V120_ASYNCH_COMP        10
#define CAPI_B2_V120_TRANSPARENT        11
#define CAPI_B2_LAPD_FREE_SAPI          12

/* B3 protocols */
#define CAPI_B3_TRANSPARENT             0     /* default */
#define CAPI_B3_T90NL                   1
#define CAPI_B3_ISO8208                 2
#define CAPI_B3_X25_DCE                 3
#define CAPI_B3_T30_FAX_G3              4
#define CAPI_B3_T30_FAX_G3_EXT          5
/*#define ?                               6*/
#define CAPI_B3_MODEM                   7



/* --- protocol configuration parameters --- */

/* parameters for B1 configuration */

/* values for b1 configuration fields */
#define CAPI_B1CFG_PARITY_NONE          0     /* default */
#define CAPI_B1CFG_PARITY_ODD           1
#define CAPI_B1CFG_PARITY_EVEN          2
#define CAPI_B1CFG_STOPBIT_ONE          0     /* default */
#define CAPI_B1CFG_STOPBIT_TWO          1
#define CAPI_B1CFG_FG3_TX_LEVEL_LOUD    0
#define CAPI_B1CFG_FG3_TX_LEVEL_MED     7
#define CAPI_B1CFG_FG3_TX_LEVEL_SILENT  15
#define CAPI_B1CFG_SPEED_NEG_NONE       0
#define CAPI_B1CFG_SPEED_NEG_MODCLASS   1
#define CAPI_B1CFG_SPEED_NEG_V100       2
#define CAPI_B1CFG_SPEED_NEG_V8         3

/* bits in B1 configuration parameter "options" */
#define CAPI_B1CFG_OPT_DIS_RETRAIN      0x0001
#define CAPI_B1CFG_OPT_DIS_RINGTONE     0x0002
#define CAPI_B1CFG_OPT_GUARDTONE_MASK   0x000C
#define CAPI_B1CFG_OPT_GUARDTONE_NONE   0x0000
#define CAPI_B1CFG_OPT_GUARDTONE_1800   0x0004
#define CAPI_B1CFG_OPT_GUARDTONE_550    0x0008
#define CAPI_B1CFG_OPT_LOUDSPKR_MASK    0x0030
#define CAPI_B1CFG_OPT_LOUDSPKR_OFF     0x0000
#define CAPI_B1CFG_OPT_LOUDSPKR_TRAINON 0x0010
#define CAPI_B1CFG_OPT_LOUDSPKR_ALLON   0x0020
#define CAPI_B1CFG_OPT_VOLUME_MASK      0x00C0
#define CAPI_B1CFG_OPT_VOLUME_SILENT    0x0000
#define CAPI_B1CFG_OPT_VOLUME_LOW       0x0040
#define CAPI_B1CFG_OPT_VOLUME_HIGH      0x0080
#define CAPI_B1CFG_OPT_VOLUME_MAX       0x00C0

/* some selected modem, V.100 and fax-g3 transmission rates */
#define CAPI_B1CFG_BITRATE_MAXIMUM      0
#define CAPI_B1CFG_BITRATE_ADAPTIVE     0
#define CAPI_B1CFG_BITRATE_2400         2400
#define CAPI_B1CFG_BITRATE_4800         4800
#define CAPI_B1CFG_BITRATE_7200         7200
#define CAPI_B1CFG_BITRATE_9600         9600
#define CAPI_B1CFG_BITRATE_12000        12000
#define CAPI_B1CFG_BITRATE_14400        14400
#define CAPI_B1CFG_BITRATE_28800        28800
#define CAPI_B1CFG_BITRATE_33600        33600
#define CAPI_B1CFG_BITRATE_57600        57600

/* B1 configuration for V.110 asynchronous operation (2) and V.110 synchronous
 * operation with HDLC framing (3)
 */
typedef struct CAPIB1ConfigV110
{
   cBYTE bLength;               /* length of structure */
   cWORD wRate;                 /* values from CAPI_B1CFG_BITRATE_*; default
                                 * CAPI_B1CFG_BITRATE_ADAPTIVE for
                                 * B1-protocol 2 and CAPI_B1CFG_BITRATE_57600
                                 * for B1-protocol 3
                                 */
   cWORD wBitsPerChar;          /* no. of bits per char., default is 8;
                                 * is reserved and coded as 0 for B1-protocol 3
                                 */
   cWORD wParity;               /* values from CAPI_B1CFG_PARITY_*;
                                 * is reserved and coded as 0 for B1-protocol 3
                                 */
   cWORD wStopbits;             /* values from CAPI_B1CFG_STOPBIT_*, should be
                                 * CAPI_B1CFG_STOPBIT_ONE; is reserved and
                                 * coded as 0 for B1-protocol 3
                                 */
} CAPIB1ConfigV110_t;

/* B1 configuration for fax-g3 operation (4) */
typedef struct CAPIB1ConfigFaxG3
{
   cBYTE bLength;               /* length of structure */
   cWORD wRate;                 /* values from CAPI_B1CFG_BITRATE_*, default
                                 * CAPI_B1CFG_BITRATE_ADAPTIVE
                                 */
   cWORD wTransmitLevel;        /* 0..15; 0 loud, 15 quiet */
   cWORD wParity;               /* values from CAPI_B1CFG_PARITY_*, should be
                                 * CAPI_B1CFG_PARITY_NONE */
   cWORD wStopbits;             /* values from CAPI_B1CFG_STOPBIT_*, should be
                                 * CAPI_B1CFG_STOPBIT_ONE
                                 */
} CAPIB1ConfigFaxG3_t;

/* B1 configuration for analog modem operation with full negotiation (7),
 * analog modem asynchronous operation (8) and analog modem synchronous
 * operation with HDLC framing (9)
 */
typedef struct CAPIB1ConfigModem
{
   cBYTE bLength;               /* length of structure */
   cWORD wRate;                 /* values from CAPI_B1CFG_BITRATE_*, default
                                 * CAPI_B1CFG_BITRATE_ADAPTIVE
                                 */
   cWORD wBitsPerChar;          /* no. of bits per char., default is 8;
                                 * is reserved and coded as 0 for B1-protocol 9
                                 */
   cWORD wParity;               /* values from CAPI_B1CFG_PARITY_*, default
                                 * CAPI_B1CFG_PARITY_NONE; is reserved and
                                 * coded as 0 for B1-protocol 9
                                 */
   cWORD wStopbits;             /* values from CAPI_B1CFG_STOPBIT_*, default
                                 * CAPI_B1CFG_STOPBIT_ONE; is reserved and
                                 * coded as 0 for B1-protocol 9
                                 */
   cWORD wOptions;              /* bit field of options from CAPI_B1CFG_OPT_*
                                 */
   cWORD wSpeedNegotiation;     /* type of speed negotiation, values from
                                 * CAPI_B1CFG_SPEED_NET_*
                                 */
} CAPIB1ConfigModem_t;



/* parameters for B2 configuration */

/* values for B2 configuration fields */
#define CAPI_B2CFG_MODULO_NORMAL        8     /* default */
#define CAPI_B2CFG_MODULO_EXTENDED      128
#define CAPI_B2CFG_MODEMOPT_DIS_V42     0x0001
#define CAPI_B2CFG_MODEMOPT_DIS_MNP     0x0002
#define CAPI_B2CFG_MODEMOPT_DIS_TRANSPARENT \
                                        0x0004
#define CAPI_B2CFG_MODEMOPT_DIS_V42NEG  0x0008
#define CAPI_B2CFG_MODEMOPT_DIS_COMP    0x0010
#define CAPI_B2CFG_V42BIS_DIR_ALL       0
#define CAPI_B2CFG_V42BIS_DIR_INCOMING  1
#define CAPI_B2CFG_V42BIS_DIR_OUTGOING  2

/* B2 configuration for ISO 7776 (X.75 SLP) (0) */
typedef struct CAPIB2ConfigX75Slp
{
   cBYTE      bLength;          /* length of structure */
   cBYTE      bAddressA;        /* link address A, default is 0x03 */
   cBYTE      bAddressB;        /* link address B, default is 0x01 */
   cBYTE      bModuloMode;      /* normal (default) or extended operation,
                                 * valuesfrom CAPI_B2CFG_MODULO_*
                                 */
   cBYTE      bWindowSize;      /* data transmit window size, default 7 */
/* CAPIStruct XID; */           /* reserved, coded as empty struct */
} CAPIB2ConfigX75Slp_t;

/* B2 configuration for SDLC (2) */
typedef struct CAPIB2ConfigSdlc
{
   cBYTE      bLength;          /* length of structure */
   cBYTE      bAddress;         /* link address, default is 0xC1 */
   cBYTE      bReserved;        /* reserved, coded as 0 */
   cBYTE      bModuloMode;      /* normal (default) or extended operation,
                                 * valuesfrom CAPI_B2CFG_MODULO_*
                                 */
   cBYTE      bWindowSize;      /* data transmit window size, default 7 */
/* CAPIStruct XID; */           /* contents of the XID response when a XID
                                 * command is received
                                 */
} CAPIB2ConfigSdlc_t;

/* B2 configuration for LAPD for D-channel X.25 (3) */
typedef struct CAPIB2ConfigLapdX25
{
   cBYTE      bLength;          /* length of structure */
   cBYTE      bAddressA;        /* type of TEI assignment and TEI value */
   cBYTE      bReserved;        /* reserved, coded as 0 */
   cBYTE      bModuloMode;      /* normal (default) or extended operation,
                                 * valuesfrom CAPI_B2CFG_MODULO_*
                                 */
   cBYTE      bWindowSize;      /* data transmit window size, default 3 */
/* CAPIStruct XID; */           /* reserved, coded as empty struct */
} CAPIB2ConfigLapdX25_t;

/* B2 configuration for modem with full negotiation (7) */
typedef struct CAPIB2ConfigModemFull
{
   cBYTE bLength;               /* length of structure */
   cWORD wOptions;              /* bit field of options for modem operation,
                                 * bits from CAPI_B2CFG_MODEMOPT_*
                                 */
} CAPIB2ConfigModemFull_t;

/* B2 configuration for V.120 asynchronous mode (9) and V.120 bit-transparent
 * mode (11)
 */
typedef struct CAPIB2ConfigV120
{
   cBYTE      bLength;          /* length of structure */
   cBYTE      bAddressA;        /* low byte of LLI, default is 0x00 */
   cBYTE      bAddressB;        /* high byte of LLI, default is 0x01 */
   cBYTE      bModuloMode;      /* extended operation, must be
				 * CAPI_B2CFG_MODULO_EXTENDED
                                 */
   cBYTE      bWindowSize;      /* data transmit window size, default 7 */
/* CAPIStruct XID; */           /* reserved, coded as empty struct */
} CAPIB2ConfigV120_t;

/* B2 configuration for ISO 7776 (X.75 SLP) modified supporting V.42bis (8)
 * and V.120 asynchronous mode with V.42bis (10)
 */
/* Note: All fields have the same meaning as in the corresponding structure
 *       without V.42bis.
 */
typedef struct CAPIB2ConfigV42bis
{
   cBYTE bLength;               /* length of structure */
   cBYTE bAddressA;
   cBYTE bAddressB;
   cBYTE bModuloMode;
   cBYTE bWindowSize;
   cWORD wDirection;            /* enable compression / decompression for some
				 * direction, one of CAPI_B2CFG_V42BIS_DIR_*
                                 */
   cWORD wNumCodeWords;         /* parameter P1 of V.42bis, default is
                                 * manufacturer dependent; a default value of
                                 * 2048 is suggested
                                 */
   cWORD wMaxStringLength;      /* parameter P2 of V.42bis, value in the range
				 * from 6 to 250, default is 250
                                 */
} CAPIB2ConfigV42bis_t;



/* parameters for B3 configuration */

/* fax-g3 resolutions for simple T.30 (B3-protocol 4) */
#define CAPI_FAXG3_RESOLUTION_STANDARD  0
#define CAPI_FAXG3_RESOLUTION_FINE      1

/* fax-g3 options parameter for extended T.30 (B3-protocol 5) */
#define CAPI_FAXG3OPT_ENABLE_HIGHRES    0x0001
#define CAPI_FAXG3OPT_FAX_POLL          0x0002
#define CAPI_FAXG3OPT_MORE_PAGES        0x0004
#define CAPI_FAXG3OPT_NO_ECM            0x8000

/* fax-g3 data format definitions */
#define CAPI_FAXG3_FORMAT_SFF           0
#define CAPI_FAXG3_FORMAT_PLAIN_FAX     1
#define CAPI_FAXG3_FORMAT_PCX           2
#define CAPI_FAXG3_FORMAT_DCX           3
#define CAPI_FAXG3_FORMAT_TIFF          4
#define CAPI_FAXG3_FORMAT_ASCII         5
#define CAPI_FAXG3_FORMAT_EXT_ANSI      6
#define CAPI_FAXG3_FORMAT_BINARY_FILE   7

/* B3 configuration for T.90NL (1), ISO 8208 (2) and X.25 DCE (3) */
typedef struct CAPIB3ConfigT90NL
{
   cBYTE bLength;               /* length of structure */
   cWORD wLIC;
   cWORD wHIC;
   cWORD wLTC;
   cWORD wHTC;
   cWORD wLOC;
   cWORD wHOC;
   cWORD wModuloMode;
   cWORD wWindowSize;
} CAPIB3ConfigT90NL_t;

/* B3 configuration for fax-g3, simple T.30 (B3-protocol 4) */
typedef struct CAPIB3ConfigFaxG3
{
   cBYTE      bLength;          /* length of structure */
   cWORD      wResolution;      /* values from FAXG3_RESOLUTION_* */
   cWORD      wFormat;          /* values from FAXG3_FORMAT_* */
/* CAPIStruct CallingStationID; */
/* CAPIStruct Headline;         */
} CAPIB3ConfigFaxG3_t;

/* B3 configuration for fax-g3, extended T.30 (B3-protocol 5) */
typedef struct CAPIB3ConfigFaxG3Ext
{
   cBYTE      bLength;          /* length of structure */
   cWORD      wOptions;         /* bit mask of FAXG3OPT_* */
   cWORD      wFormat;          /* values from FAXG3_FORMAT_* */
/* CAPIStruct CallingStationID; */
/* CAPIStruct Headline;         */
} CAPIB3ConfigFaxG3Ext_t;



/* parameters for B-channel operation */

/* B-channel operation modes */
#define CAPI_BPROT_GLOBOPT_BCHNOP_DEFAULT       0
#define CAPI_BPROT_GLOBOPT_BCHNOP_DTE           1
#define CAPI_BPROT_GLOBOPT_BCHNOP_DCE           2

/* global options for setting of B-channel operation */
typedef struct CAPIBProtocolGlobalOptions
{
   cBYTE bLength;               /* length of structure */
   cWORD wBChannelOperation;    /* operation mode of the B-channel,
                                 * values from
                                 * CAPI_BPROT_GLOBOPT_BCHOP_*
                                 */
} CAPIBProtocolGlobalOptions_t;



/* accumulated structure for the B-channel protocol */
typedef struct CAPIBProtocol
{
   cBYTE      bLength;          /* length of structure */
   cWORD      wB1protocol;      /* B-chn. layer 1 protocol, values from
                                 * CAPI_B1_*
                                 */
   cWORD      wB2protocol;      /* B-chn. layer 2 protocol, values from
                                 * CAPI_B2_*
                                 */
   cWORD      wB3protocol;      /* B-chn. layer 3 protocol, values from
                                 * CAPI_B3_*
                                 */
/* CAPIStruct b1config;         */
/* CAPIStruct b2config;         */
/* CAPIStruct b3config;         */
/* CAPIStruct globalConfig;     */
} CAPIBProtocol_t;



/* --- definition of the NCPI for various protocols --- */

/* standard NCPI (ISO8208, X.25) */
typedef struct CAPIStdNCPI
{
   cBYTE bLength;                       /* length of structure */
   cBYTE b0;
   cBYTE b1;
   cBYTE b2;
/* cBYTE bx []; */
} CAPIStdNCPI_t;

/* NCPI for fax-g3, simple T.30 */
typedef struct CAPIFaxG3NCPI
{
   cBYTE      bLength;                  /* length of structure */
   cWORD      wSpeed;
   cWORD      wResolution;
   cWORD      wFormat;
   cWORD      wPages;
/* CAPIStruct receive_id;               */
} CAPIFaxG3NCPI_t;

/* NCPI for fax-g3, extended T.30 */
typedef struct CAPIFaxG3ExtNCPI
{
   cBYTE      bLength;                  /* length of structure */
   cWORD      wSpeed;
   cWORD      wOptions;
   cWORD      wFormat;
   cWORD      wPages;
/* CAPIStruct receive_id; */
} CAPIFaxG3ExtNCPI_t;

/* NCPI for modem operation */
typedef struct CAPIModemNCPI
{
   cBYTE bLength;                       /* length of structure */
   cWORD wSpeed;
   cWORD wProtocol;
} CAPIModemNCPI_t;

/* union of all NCPI types */
typedef union CAPINCPI
{
   struct CAPIStdNCPI      std;
   struct CAPIFaxG3NCPI    fax;
   struct CAPIFaxG3ExtNCPI faxExt;
   struct CAPIModemNCPI    modem;
} CAPINCPI_t;



/* --- CIP values & masks --- */

#define CAPI_CIP_NO                             0
#define CAPI_CIP_SPEECH                         1
#define CAPI_CIP_UNRESTRICTED_DATA              2
#define CAPI_CIP_RESTRICTED_DATA                3
#define CAPI_CIP_3100Hz_AUDIO                   4
#define CAPI_CIP_7kHz_AUDIO                     5
#define CAPI_CIP_VIDEO                          6
#define CAPI_CIP_PACKET_MODE                    7
#define CAPI_CIP_DATA_56                        8
#define CAPI_CIP_UNRESTRICTED_DATA_TONES        9
#define CAPI_CIP_TELEPHONY                      16
#define CAPI_CIP_FAX_G2_G3                      17
#define CAPI_CIP_FAX_G4_CLASS1                  18
#define CAPI_CIP_TELETEX_BASIC_MIXED            19
#define CAPI_CIP_FAX_G4_CLASS2_CLASS3           19
#define CAPI_CIP_TELETEX_BASIC_PROCESSABLE      20
#define CAPI_CIP_TELETEX_BASIC                  21
#define CAPI_CIP_VIDEOTEX                       22
#define CAPI_CIP_TELEX                          23
#define CAPI_CIP_X400                           24
#define CAPI_CIP_X200                           25
#define CAPI_CIP_7kHz_TELEPHONY                 26
#define CAPI_CIP_VIDEO_TELEPHONY_1ST            27
#define CAPI_CIP_VIDEO_TELEPHONY_2ND            28

#define CAPI_CIP_MASK_UNLISTEN                  0x00000000UL
#define CAPI_CIP_MASK_ANY                       0x00000001UL
#define CAPI_CIP_MASK_SPEECH                    0x00000002UL
#define CAPI_CIP_MASK_UNRESTRICTED_DATA         0x00000004UL
#define CAPI_CIP_MASK_RESTRICTED_DATA           0x00000008UL
#define CAPI_CIP_MASK_3100Hz_AUDIO              0x00000010UL
#define CAPI_CIP_MASK_7kHz_AUDIO                0x00000020UL
#define CAPI_CIP_MASK_VIDEO                     0x00000040UL
#define CAPI_CIP_MASK_PACKET_MODE               0x00000080UL
#define CAPI_CIP_MASK_DATA_56                   0x00000100UL
#define CAPI_CIP_MASK_UNRESTR_DATA_TONES        0x00000200UL
#define CAPI_CIP_MASK_TELEPHONY                 0x00010000UL
#define CAPI_CIP_MASK_FAX_G2_G3                 0x00020000UL
#define CAPI_CIP_MASK_FAX_G4_CLASS1             0x00040000UL
#define CAPI_CIP_MASK_TELETEX_BASIC_MIXED       0x00080000UL
#define CAPI_CIP_MASK_FAX_G4_CLASS2_CLASS3      0x00080000UL
#define CAPI_CIP_MASK_TELETEX_BASIC_PROCESS     0x00100000UL
#define CAPI_CIP_MASK_TELETEX_BASIC             0x00200000UL
#define CAPI_CIP_MASK_VIDEOTEX                  0x00400000UL
#define CAPI_CIP_MASK_TELEX                     0x00800000UL
#define CAPI_CIP_MASK_X400                      0x01000000UL
#define CAPI_CIP_MASK_X200                      0x02000000UL
#define CAPI_CIP_MASK_7kHz_TELEPHONY            0x04000000UL
#define CAPI_CIP_MASK_VIDEO_TELEPHONY_1ST       0x08000000UL
#define CAPI_CIP_MASK_VIDEO_TELEPHONY_SND       0x10000000UL

#define CAPI_CIP_MASK2_UNLISTEN                 0x00000000UL



/* --- listen info mask --- */

#define CAPI_INFO_MASK_NONE                     0x00000000UL
#define CAPI_INFO_MASK_CAUSE                    0x00000001UL
#define CAPI_INFO_MASK_DATE_TIME                0x00000002UL
#define CAPI_INFO_MASK_DISPLAY                  0x00000004UL
#define CAPI_INFO_MASK_USER_USER                0x00000008UL
#define CAPI_INFO_MASK_CALL_PROGRESSION         0x00000010UL
#define CAPI_INFO_MASK_FACILITY                 0x00000020UL
#define CAPI_INFO_MASK_CHARGING                 0x00000040UL
#define CAPI_INFO_MASK_CALLED_PARTY_NUMBER      0x00000080UL
#define CAPI_INFO_MASK_CHANNEL_ID               0x00000100UL
#define CAPI_INFO_MASK_EARLY_B3_CONNECT         0x00000200UL
#define CAPI_INFO_MASK_REDIRECTION_INFO         0x00000400UL
#define CAPI_INFO_MASK_SENDING_COMPLETE         0x00001000UL

#define CAPI_INFO_MASK_ALL                      0x000017FFUL



/* --- data block flags --- */

#define CAPI_DATA_B3_FLAG_QUALIFIER     0x0001
#define CAPI_DATA_B3_FLAG_MORE_DATA     0x0002
#define CAPI_DATA_B3_FLAG_DELIVERY_CONF 0x0004
#define CAPI_DATA_B3_FLAG_EXPEDITED     0x0008
#define CAPI_DATA_B3_FLAG_UI_FRAME      0x0010
#define CAPI_DATA_B3_FLAG_BREAK         0x0010
#define CAPI_DATA_B3_FLAG_FRAME_ERROR   0x8000



/* --- call reject values --- */

#define CAPI_CALL_ACCEPT                        0
#define CAPI_CALL_IGNORE                        1
#define CAPI_CALL_REJECT_NORMAL_CLEARING        2
#define CAPI_CALL_REJECT_USER_BUSY              3
#define CAPI_CALL_REJECT_NO_CHANNEL             4
#define CAPI_CALL_REJECT_FACILITY               5
#define CAPI_CALL_REJECT_CHANNEL_UNACCEPTABLE   6
#define CAPI_CALL_REJECT_INCOMPAT_DESTINATION   7
#define CAPI_CALL_REJECT_OUT_OF_ORDER           8



/* --- facilitiy parameters --- */

/* facility selectors */
#define CAPI_FACILITY_SELECTOR_HANDSET                  0
#define CAPI_FACILITY_SELECTOR_DTMF                     1
#define CAPI_FACILITY_SELECTOR_V42BIS                   2
#define CAPI_FACILITY_SELECTOR_SUPPL_SERVICES           3
#define CAPI_FACILITY_SELECTOR_POWER_MNGMNT_WAKEUP      4
#define CAPI_FACILITY_SELECTOR_LINE_INTERCONNECT        5
#define CAPI_FACILITY_SELECTOR_BROADBAND_EXTENSIONS     6
#define CAPI_FACILITY_SELECTOR_CONTROLLER_EVENTS        7
#define CAPI_FACILITY_SELECTOR_ECHO_CANCELLATION        8



/* --- definitions for facility DTMF --- */

/* DTMF functions */
#define CAPI_DTMF_START_LISTEN  1
#define CAPI_DTMF_STOP_LISTEN   2
#define CAPI_DTMF_SEND_DIGITS   3

/* structure of a DTMF request */
typedef struct CAPIFacilityDtmfReq
{
   cBYTE      bLength;
   cWORD      wFunction;
   cWORD      wToneDuration;
   cWORD      wGapDuration;
/* CAPIStruct Digits;                   */
} CAPIFacilityDtmfReq_t;

/* structure of a DTMF confirmation */
typedef struct CAPIFacilityDtmfConf
{
   cBYTE bLength;
   cWORD wInfo;
} CAPIFacilityDtmfConf_t;

/* structure of a DTMF indication */
/* typedef struct CAPIFacilityDtmfInd*/
/* {                                    */
/*    cBYTE Digits [];                  */
/* } CAPIFacilityDtmfInd_t;*/

/* Struktur einer DTMF-Response */
/* typedef struct CAPIFacilityDtmfResp  */
/* {                                    */
/* } CAPIFacilityDtmfResp_t; */



/* --- CAPI profile definitions --- */

/* structure of a adapter profile */
typedef struct CAPIProfileBuffer
{
   cWORD  wCtlr;                        /* number of installed controllers */
   cWORD  wNumBChannels;                /* number of supported B-channels */
   cDWORD dwGlobalOptions;              /* bit field for global options,
                                         * combination of CAPI_PROFILE_* values
                                         */
   cDWORD dwB1ProtocolSupport;          /* bit field of supported B1 protocols,
                                         * combination of values from CAPI_B1_*
                                         */
   cDWORD dwB2ProtocolSupport;          /* bit field of supported B2 protocols,
                                         * combination of values from CAPI_B2_*
                                         */
   cDWORD dwB3ProtocolSupport;          /* bit field of supported B3 protocols,
                                         * combination of values from CAPI_B3_*
                                         */
   cBYTE  bReserved [24];               /* reserved for (future) CAPI use */
   cBYTE  bManufacturerSpecific [20];   /* manufacturer specific information */
} CAPIProfileBuffer_t;

/* Global options */
#define CAPI_PROFILE_INTERNAL_CTLR_SUPPORT      0x0001
#define CAPI_PROFILE_EXTERNAL_EQUIPMENT_SUPPORT 0x0002
#define CAPI_PROFILE_HANDSET_SUPPORT            0x0004
#define CAPI_PROFILE_DTMF_SUPPORT               0x0008
#define CAPI_PROFILE_SUPPLEMENTARY_SERVICES     0x0010
#define CAPI_PROFILE_CHANNEL_ALLOCATION         0x0020
#define CAPI_PROFILE_BCHANNEL_OPERATION         0x0040
#define CAPI_PROFILE_LINE_INTERCONNECT          0x0080
#define CAPI_PROFILE_BROADBAND_EXTENSIONS       0x0100
#define CAPI_PROFILE_ECHO_CANCELLATION          0x0200



/* --- Definitions for the B-channel information element --- */

/* the channel mode values */
#define CAPI_BCHAN_MODE_B               0
#define CAPI_BCHAN_MODE_D               1
#define CAPI_BCHAN_MODE_NONE            2
#define CAPI_BCHAN_MODE_LEASED_ALLOC    3
#define CAPI_BCHAN_MODE_CHANNEL_ID      4

/* the sub-structure for leased line channel allocation */
typedef struct CAPIBChannelInfoAllocation
{
   cWORD wOperation;
   cBYTE abChannelMaskArray [31];
} CAPIBChannelInfoAllocation_t;

/* the structure for all B-channel information types */
typedef struct CAPIBChannelInfo
{
   cBYTE bLength;               /* length of structure */
   cWORD wChannel;              /* channel mode, one of CAPI_BCHAN_MODE_* */
   union
   {
      struct CAPIBChannelInfoAllocation alloc;
/*    CAPIStruct                        chanID; */
   } info;
} CAPIBChannelInfo_t;



/* --- Miscellaneous definitions --- */

/* mode values for Sending Complete support */
#define CAPI_SENDING_COMPLETE_NO        0
#define CAPI_SENDING_COMPLETE_YES       1

/* structure of a Sending Complete specification */
typedef struct CAPISendingComplete
{
   cBYTE bLength;               /* length of structure */
   cWORD wMode;                 /* mode for sending complete support, one of
				 * CAPI_SENDING_COMPLETE_*
                                 */
} CAPISendingComplete_t;



#include <capi_pack_off.h>





/* === prototypes of interface functions ================================= */





#endif /* __CAPI20_H */
