/**
 * @file pack_on.h
 *
 * CAPI-Pack-On - Set the packing of structures in the CAPI message context to
 * byte boundaries.
 *
 * Copyright:   2002-2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: capi_pack_on.h,v 1.11.2.1 2005/05/27 16:29:07 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     CAPI headers $
 * @date        03.08.2002
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * @section description Description
 *
 * By including this header file the setting for the alignment of structure
 * members will be forced to byte boundaries, i.e. the highest possible
 * packing density will be selected. This is the smallest common denominator for
 * all compilers on all (known or relevant) platforms and so the exchange of
 * data in binary form is possible (for limited purposes).
 *
 * It is very important to include <capi_pack_off.h> after this header file if
 * this pair is used within other header files. Else other structure will also
 * be compiled with byte packing. But as other modules might not include header
 * files in the same order, some structure may be compiled with byte packing in
 * one module and with default packing in the other. This will lead to very hard
 * to find runtime errors.
 *
 * According to the current platform and the compiler package used the correct
 * commands must be used:
 *
 * @li Microsoft Visual C++ v6.0: Supports "#pragma pack(x)" as well as
 *        "#pragma pack(push|pop)". Because of the the usage in pairs of
 *        <capi_pack_on.h> and <capi_pack_off.h> the push/pop method is chosen.
 *
 * @li Borland: Supports at least since BC v4.5 "#pragma option -a-". If
 *        "#pragma pack(x)" is also supported should be checked if needed.
 *
 * @li Watcom: Supports "#pragma pack(x)".
 *
 * @li GNU-C (at least since v2.95): Supports "#pragma pack(x)" and special
 *        attributes within the structure declarations ("__attribute((pack))").
 *        The later has the disadvantage, that the corresponding structure
 *        cannot be documented by doxygen any more, because the attribute will
 *        falsely be recognized as the structure name. So for GNU-C "#pragma
 *        pack(x)" will be used, too.
 *
 * </p>
 */

#if defined(__BORLANDC__)

#pragma nopackwarning
#pragma option -a-
#define PACKED_STRUCTURE

#elif (defined(__WATCOMC__) || defined(_MSC_VER) || defined(__C89_SINIX_Z__) || defined (__C89_SINIX_N__))

#if (defined(_MSC_VER) && defined(WIN32))

#if ( _MSC_VER >= 800 ) || defined(_PUSHPOP_SUPPORTED)
#pragma warning(disable:4103)
#endif

#pragma pack(push, 1)
#define PACKED_STRUCTURE

#else

#pragma pack(1)
#define PACKED_STRUCTURE

#endif

#elif defined(__GNUC__)

/* GNU-C at least since v2.95 knows of the pack pragmas,
 * __attribute__((packed)) is not needed any more.
 */
/*#define PACKED_STRUCTURE __attribute__((packed))*/
#pragma pack(1)
#define PACKED_STRUCTURE

#elif defined(TARGET_AIX)

#pragma options align=packed
#define PACKED_STRUCTURE

#elif defined(TARGET_DEC)

#pragma pack(1)
#define PACKED_STRUCTURE

#elif defined(TARGET_HPUX)

#pragma HP_ALIGN NOPADDING PUSH
#define PACKED_STRUCTURE

#else
#error Unknown compiler, check "capi_pack_on.h" and "capi_pack_off.h"
#endif

