/**
 * @file ix1a_board_params.h
 *
 * IX1A-BoardParameters - Board parameter definitions for IX1A ISDN adapters.
 *
 * Copyright: 2005 Thomas Wintergerst. All rights reserved.
 *
 * $Id: ix1a_board_params.h,v 1.4.2.1 2005/05/27 16:29:10 thomas Exp $
 * $Project:    CAPI for BSD $
 * $Target:     ix1a - CAPI manager driver for IX1 active ISDN controllers $
 * @date        09.01.2005
 * @author      "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __IX1A_BOARD_PARAMS_H
#define __IX1A_BOARD_PARAMS_H

#include <sys/cdefs.h>          /* __packed */
#include <sys/types.h>          /* u_int32_t, u_int8_t, etc. */





/* === Public declarations =============================================== */





/** The version of the board configuration data structure. */
#define IX1A_BOARD_CONFIG_VERSION       1

/** The maximum number of line ports supported by any board. */
#define IX1A_MAX_PORTS                  4

/** The number of EAZ mapping entries. */
#define IX1A_EAZ_MAPPING_COUNT          10

/**
 * The length of one EAZ mapping MSN.
 *
 * This is the full aligned length of the parameter string field. Only 8 digits
 * plus terminating null byte may really be used.
 */
#define IX1A_EAZ_MAPPING_MSN_LEN        12

/** The length of SPIDs for north american D-channel protocols. */
#define IX1A_SPID_LEN                   20

/**
 * The number of configurable DN numbers for north american D-channel
 * protocols.
 */
#define IX1A_DN_COUNT                   5

/** The length of DN number for north american D-channel protocols. */
#define IX1A_DN_LEN                     16



/** General board parameters. */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
   u_int32_t dwMvipType;        /**<
                                 * Operation type of the MVIP bus, if the
                                 * board owns one (Octo, Primary, Multimodem).
                                 * 1 - board is master, 2 - board is slave.
                                 * Leave unset or 0 if the board does not have
                                 * an MVIP bus or it is not used.
                                 */
   u_int32_t dwCodingLaw;       /**<
                                 * Coding law for raw B-channel data. 1 - alaw
                                 * (most parts of the world), 2 - law (north
                                 * america, Japan). Leave unset or 0 to use
                                 * default setting.
                                 */
   u_int32_t dwMultimodemCountryCode;
                                /**<
                                 * Only for Multimodem boards: A value other
                                 * than 0 optimises the board's modem chips for
                                 * a specific country. See the Rockwell AT
                                 * Reference Manual for possible values.
                                 */
} Ix1aBoardGeneralParams_t;

/** Fax board parameters. */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
} Ix1aBoardFaxParams_t;

/** Modem board parameters. */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
} Ix1aBoardModemParams_t;

/** Compression board parameters. */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
   u_int32_t dwCompMode;        /**<
                                 * The compression to use for normal (X.75)
                                 * connections without application control.
                                 * Leave it set to null for default operation.
                                 * Compression can be forced to one of the
                                 * following modes: 1 - none, 2 - V42bis, 3 -
                                 * ITK-specific compression, 4 - optional
                                 * (boards on both sides detect each other and
                                 * choose V.42bis or ITK), 5 - CAPIPlus
                                 * defined (applications can select through
                                 * CAPIPlus messages, not supported by c4b).
                                 */
} Ix1aBoardCompParams_t;

/** Line interface parameters. */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
   u_int32_t dwLineAccess;      /**<
                                 * 0 - default setting from the board's EEPROM
                                 * (discouraged), 1 - multipoint (MSN lines),
                                 * 2 - point-to-point.
                                 */
   u_int32_t dwLineType;        /**<
                                 * 0 - default setting from the board's EEPROM
                                 * (discouraged), 1 - switched line, 2 - leased
                                 * line.
                                 */
   u_int32_t dwLeasedLineType;  /**<
                                 * Only for leased lines: 1 - single B-channel
                                 * with D-channel, 2 - single B-channel without
                                 * D-channel, 3 - bundled B-channels with
                                 * D-channel, 4 - bundled B-channels without
                                 * D-channel.
                                 */
   u_int32_t dwNtTeSide;        /**<
                                 * Set the board to operate as a terminal
                                 * device or to have it emulate the network
                                 * side. 0 - default setting from the board's
                                 * EEPROM (discouraged), 1 - TE side, 2 - NT
                                 * side.
                                 */
   u_int32_t dwClockGenerator;  /**<
                                 * Primary only: Use clock from line (normal
                                 * operation) or provide clocking to the line
                                 * (used for NT emulation). 0 - default setting
                                 * from the board's EEPROM (discouraged), 1 -
                                 * clock from line, 2 - clock from board.
                                 */
   u_int32_t dwLineBitrate;     /**<
                                 * Primary only: Set the board to E1 or T1
                                 * mode. 0 - default setting from the board's
                                 * EEPROM (discouraged), 1 - E1 (30
                                 * B-channels), 2 - T1 (23 B-channels).
                                 */
   u_int32_t dwFirstBChan;      /**<
                                 * Primary only: The timeslot value for the
                                 * lowest B-channel to use. 0 - no limit, 1-30
                                 * - the B-channel's timeslot number.
                                 */
   u_int32_t dwLastBChan;       /**<
                                 * Primary only: The timeslot value for the
                                 * highest B-channel to use. 0 - no limit, 1-30
                                 * - the B-channels timeslot number.
                                 */
   u_int32_t dwBChanMask;       /**< Currently not used. */
   u_int32_t dwBChanSelection;  /**<
                                 * Primary only: Algorithm to use for selecting
                                 * the next B-channel to use. 0 - default
                                 * setting from the board's EEPROM, 1 - round
                                 * robin up, 2 - round robin down, 3 - fixed
                                 * up, 4 - fixed down.
                                 */
   u_int32_t dwLineCode;        /**<
                                 * Primary only: Data coding on the line. 0 -
                                 * default setting for the D-channel protocol,
                                 * 1 - B8ZS (T1), 2 - AMI with ZCS (T1), 3 -
                                 * HDB3 (E1), 4 - AMI (E1).
                                 */
   u_int32_t dwFramingFormat;   /**<
                                 * Primary only: The framing format to use on
                                 * the line. 0 - default setting for the
                                 * D-channel protocol, 1 - ESF (T1), 2 - SF
                                 * (D3/D4) (T1), 3 - F4 (T1), 4 - F72 (T1), 5 -
                                 * CRC4 multi-frame (E1), 6 - double frame
                                 * (E1).
                                 */
   u_int32_t dwDsxPreEmphasis;  /**<
                                 * Primary and T1 only: DSX pre-emphasis. 0 -
                                 * default value, 1 - 0-115 feet, 2 - 82-213
                                 * feet, 3 - 180-312 feet, 4 - 279-410 feet, 5
                                 * - 377-509 feet, 6 - 476-607 feet, 7 -
                                 * 574-689 feet, 8 - 18 dB.
                                 */
} Ix1aBoardLineInterfaceParams_t;

/** D-channel parameters. */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
   u_int32_t dwDProtocol;       /**<
                                 * The D-channel protocol to use.
                                 * 0 - default setting from the board's EEPROM
                                 * (discouraged),
                                 * 1 - 1TR6,
                                 * 2 - DSS1 (Euro-ISDN, NET3, ETSI),
                                 * 3 - VN2 (france),
                                 * 4 - VN3 (france),
                                 * 5 - 5ESS (AT&T / USA),
                                 * 6 - DSS1 PBX (with SETUP-ACK),
                                 * 7 - DMS-100 (USA),
                                 * 8 - CORNET-N (Siemens PBX),
                                 * 9 - DSS1 (Norway phase 1),
                                 * 10 - CORNET-T (Siemens PBX),
                                 * 11 - National ISDN 1 (NI1, USA),
                                 * 12 - Q.931 Blue Book (CCITT).
                                 * 13 - Q.Sig,
                                 * 15 - TS013 / TS014 (Australia),
                                 * 16 - DSS1 for Netherlands (with
                                 *      type=national),
                                 * 17 - DSS1 PBX with no sending complete,
                                 * 21 - DSS1 with no RESTART,
                                 * 22 - 4ESS (USA, AT&T - T1 long distance).
                                 */
   u_int32_t dwSignalingMode;   /**<
                                 * Only common channel signaling (CCS) is
                                 * supported, channel associated signaling
                                 * (CAS) is not. 0 - default settint from the
                                 * board's EEPROM, 1 - CCS.
                                 */
   u_int32_t dwTeiType;         /**<
                                 * 0 - default setting from the board's EEPROM
                                 * (discouraged), 1 - automatic TEI assignment
                                 * (multipoint lines), 2 - fixed TEI assignment
                                 * (point-to-point lines).
                                 */
   u_int32_t dwTeiValue;        /**<
                                 * Only for fixed TEI assignment: The TEI value
                                 * to use. Possible values are 0-63, 0 is
                                 * commonly used.
                                 */
   u_int8_t  aszEazMappingTable [IX1A_EAZ_MAPPING_COUNT]
                                [IX1A_EAZ_MAPPING_MSN_LEN];
                                /**<
                                 * This array field makes only sense for CAPI
                                 * 1.1 environments, where MSNs must be
                                 * translated to EAZ numbers to be passed
                                 * through the application interface. But as
                                 * CAPI for BSD only supports CAPI 2.0 there is
                                 * no need for such an EAZ mapping.
                                 */
} Ix1aBoardDChannelParams_t;

/**
 * Service profile parameters.
 *
 * These settings are only valid for north american D-channel protocols and for
 * S0 lines. People using these protocols should known what they mean and how
 * to set them correctly. Leave the parameters unset (i.e. set to null) for
 * Primary boards.
 */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
   u_int32_t dwSignalDN2AsPrimaryDN1;
                                /**< 1 - yes, 0 - no. */
   u_int32_t dwSignalAddDN1AsPrimaryDN1;
                                /**< 1 - yes, 0 - no. */
   u_int32_t dwSignalAddDN2AsPrimaryDN2;
                                /**< 1 - yes, 0 - no. */
   u_int32_t dwSpid1DataOn;     /**< 1 - yes, 0 - no. */
   u_int32_t dwSpid2DataOn;     /**< 1 - yes, 0 - no. */
   u_int32_t dwSpid1VoiceOn;    /**< 1 - yes, 0 - no. */
   u_int32_t dwSpid2VoiceOn;    /**< 1 - yes, 0 - no. */
   u_int32_t dwMaxDnCount;      /**<
                                 * Array count of aszDn1 and aszDn2, must be
                                 * set to IX1A_DN_COUNT.
                                 */
   u_int8_t  szSpid1 [IX1A_SPID_LEN];
                                /**< The SPID for the 1st B-channel. */
   u_int8_t  szSpid2 [IX1A_SPID_LEN];
                                /**< The SPID for the 2nd B-channel. */
   u_int8_t  aszDn1 [IX1A_DN_COUNT] [IX1A_DN_LEN];
                                /**<
                                 * Array of the primary (1st entry) and
                                 * additional directory numbers for the 1st
                                 * B-channel.
                                 */
   u_int8_t  aszDn2 [IX1A_DN_COUNT] [IX1A_DN_LEN];
                                /**<
                                 * Array of the primary (1st entry) and
                                 * additional directory numbers for the 2nd
                                 * B-channel.
                                 */
} Ix1aBoardServiceProfileParams_t;

/**
 * Security Access Services parameters.
 *
 * These settings are currently not used. They would require to use CAPIPlus,
 * an extension for CAPI 1.1 from ITK. As the kernel CAPI manager of CAPI for
 * BSD does only support plain CAPI 2.0 messages, the SAS feature cannot be
 * used. So leave all fields (except the structure size) set to null.
 */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
   u_int32_t dwStateMask;
   u_int32_t dwEventMask;
   u_int32_t dwDesMode;
   u_int32_t dwHashMode;
   u_int32_t dwAutoChgSessKey;
   u_int32_t dwAutoSign;
} Ix1aBoardSasParams_t;

/**
 * Line configuration parameters.
 *
 * Each port of an IX1 active board needs an instance of this data record. They
 * must be appended to the combined parameter block as an array. Octo boards
 * need four records, Multimodem boards have no lines and thus do not expect any
 * such record. All other boards have one port and need one record instance.
 *
 * For Octo boards (those with more than only one port) each port may be
 * configured for different line settings. But it is normally a good idea to
 * operate all ports with the same settings, and even at the same S0 board of a
 * PBX or switch (clocking).
 */
typedef struct
{
   u_int32_t dwSize;            /**< Size of this structure in bytes. */
   
   Ix1aBoardLineInterfaceParams_t  interface;
   Ix1aBoardDChannelParams_t       dchan;
   Ix1aBoardServiceProfileParams_t serviceProfile;
   Ix1aBoardSasParams_t            sas;
} Ix1aBoardLineConfigParameters_t;

/**
 * The full board parameters for sending to a IX1 active board after download
 * to configure its behaviour.
 *
 * This structure contains all parameter data for configuring the firmware of a
 * IX1 active board, including the line configuration parameters. It does not
 * matter how many line ports a board has. The structure must always include
 * a full set of four line configuration parameter records. How many of them are
 * really filled is declared by setting the member dwNumLines. A Multimodem
 * board does not have any line ports, so the number of lines is set to null,
 * even if the complete structure with four line parameter records is passed to
 * the board. An Octo board has four line ports, so it needs all four line
 * configuration parameter structures to be filled correctly. All other
 * (supported) boards have a single line port and so require only a single line
 * configuration parameter structure to be filled.
 */
typedef struct
{
   u_int32_t dwVersion;
   u_int32_t dwSize;            /**<
                                 * Size of this structure in bytes, including
                                 * possibly following line configuration
                                 * parameter structures.
                                 */
   
   Ix1aBoardGeneralParams_t general;
                                /**< General board parameters. */
   Ix1aBoardFaxParams_t     fax;/**< Fax parameters. */
   Ix1aBoardModemParams_t   modem;
                                /**< Modem parameters. */
   Ix1aBoardCompParams_t    comp;
                                /**< Compression parameters. */
   
   u_int32_t dwNumLines;        /**<
                                 * The number of line ports of the board. This
                                 * field also determines how many line
                                 * configuration parameter structures follow
                                 * this structure.
                                 */
   
   /* line port parameters for the board; only as many as declared in dwNumLines
    * must really be filled (null for Multimodems, four for Octo boards, only
    * one for all others)
    */
   Ix1aBoardLineConfigParameters_t aLineParams [IX1A_MAX_PORTS];
   
} __packed Ix1aBoardParameters_t;





/* === Prototypes of interface functions ================================= */





#endif /* __IX1A_BOARD_PARAMS_H */
