/**
 * @file kcapi_ctlr.c
 *
 * KCapi-Controller - kernel CAPI manager for *BSD: internal interface for CAPI
 * controller handling.
 *
 * Copyright: 2000-2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: kcapi_ctlr.h,v 1.8.4.1 2005/05/27 16:29:13 thomas Exp $
 * Project  CAPI for BSD
 * Target   kcapimgr - The CAPI manager for *BSD
 * @date    01.01.2000
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __KCAPI_CTLR_H
#define __KCAPI_CTLR_H





/* === public declarations =============================================== */





/* === prototypes of interface functions ================================= */





/* --- initialization and close of this module --- */

/**
 * Initialize the controller handling module.
 */
extern void kcapictlr_init (void);

/**
 * Close the controller handling module.
 */
extern void kcapictlr_close (void);



/* --- controller specific part of CAPI calls --- */

/**
 * Perform application registration at a controller.
 */
extern unsigned kcapictlr_register_app
   (unsigned           uApplID,
    unsigned           uUniqueCtlrNum,
    unsigned           uMaxLogicalConnections,
    unsigned           uMaxBDataBlocks,
    unsigned           uMaxBDataLen);

/**
 * Call release of every controller for an application.
 */
extern unsigned kcapictlr_release_app
   (unsigned uApplID);

/**
 * Send a CAPI message to a controller.
 */
extern unsigned kcapictlr_put_message
   (unsigned     uApplID,
    unsigned     uCtlr,
    struct mbuf *pmbMsg);



/* --- helper functions for the CAPI trace feature --- */

/**
 * Allocate a new mbuf for a trace message.
 *
 * @note Mbufs allocated with this function should be released by calling
 *       kcapi_free_mbuf().
 *
 * @attention Mbufs of this type must not be used for normal data or CAPI
 *            messages! They are only intended for use as trace messages. The
 *            mbufs use a different type than MT_DATA to be distinguishable from
 *            normal CAPI message mbufs.
 *
 * @param nLen                  I: The size of the new mbuf in bytes.
 *
 * @return NULL                 Unable to allocate an mbuf, out of mbufs.
 * @return Else                 A pointer to the allocated mbuf.
 */
extern struct mbuf *kcapictlr_get_trace_mbuf
   (size_t nLen);

/**
 * Enqueue a trace message for later processing by the CAPI manager thread.
 *
 * This function is called by the driver level CAPI trace functions to enqueue a
 * trace message. As the driver level CAPI functions are normally called when
 * there are locks held for the CAPI application array or the controller array,
 * it is not possible to directly enqueue trace messages into the application
 * queues. So they are first enqueued into the queue for messages from
 * controllers and then passed to the designated application queue by the CAPI
 * manager thread.
 *
 * @attention The trace message mbuf must be allocated by calling
 *            kcapictlr_get_trace_mbuf()! It uses a different type than MT_DATA
 *            to distinguish trace messages from normal CAPI messages.
 *
 * @param uUniqueApplID         I: The unique application id for the application
 *                                 to receive the trace message.
 * @param pmbMsg                I: The address of an mbuf containing the trace
 *                                 message, must be allocated by
 *                                 kcapictlr_get_trace_mbuf().
 *
 * @return Nothing.
 */
extern void kcapictlr_enqueue_trace_message
   (unsigned     uUniqueApplID,
    struct mbuf *pmbMsg);





/* === definition of inline functions ==================================== */





#endif /* __KCAPI_CTLR_H */
