/**
 * @file kcapi_trace.h
 *
 * KCapi-Trace - kernel CAPI manager for *BSD: internal interface for CAPI tracing.
 *
 * Copyright: 2000-2003 Thomas Wintergerst. All rights reserved.
 *
 * $FreeBSD$
 * $Id: kcapi_trace.h,v 1.6.2.1 2005/05/27 16:29:14 thomas Exp $
 * Project  CAPI for BSD
 * Target   kcapimgr - The CAPI manager for *BSD
 * @date    01.01.2000
 * @author  "Thomas Wintergerst" <twinterg@gmx.de>
 * <p>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * </p>
 */

#ifndef __KCAPI_TRACE_H
#define __KCAPI_TRACE_H





/* === public declarations =============================================== */





/* === prototypes of interface functions ================================= */





/* --- initialization and close of this module --- */

/**
 * Initialize the trace support module.
 */
extern void kcapitrace_init (void);

/**
 * Close the trace support module.
 */
extern void kcapitrace_close (void);



/* --- registration and administration of tracers --- */

/**
 * Add a new tracer.
 *
 * @attention The caller must ensure that no call to register or release another
 *            application or tracer can interrupt the update process. This can
 *            only be accomplished by letting the CAPI application array access
 *            be blocked during this call. The implementation of this function
 *            will never call back into the application handling module, so this
 *            locking will not leed to a deadlock.
 *
 * @param uApplID               I: The application id to register the new tracer
 *                                 for.
 * @param ulTraceMask           I: The bit mask of enabled tracing features, a
 *                                 combination of C4BTRC_TRCMSK_*.
 * @param ulDataBlockLength     I: If the trace mask specifies to log Data-B3
 *                                 messages and the data blocks themselves, this
 *                                 parameter specifies the maximum number of
 *                                 bytes to trace for one data block.
 * @param fIsApplTracer         I: TRUE: The tracer is an application tracer,
 *                                    the array pafMayTraceAppl contains a flag
 *                                    for each currently registered application
 *                                    that this tracer receives messages for
 *                                    (i.e. this tracer has the right to trace),
 *                                 FALSE: The tracer is a controller tracer, the
 *                                    controller number to trace is in uCtlrNum.
 * @param pafMayTraceAppl       I: If the tracer to add is an application
 *                                 tracer, this array contains a flag for each
 *                                 registered application this tracer may
 *                                 receive trace messages from. The application
 *                                 id is used as an index into this array. The
 *                                 array should be created by checking for each
 *                                 currently registered application that the
 *                                 process owner of new tracer is either the
 *                                 owner of the application process or he is a
 *                                 member of the applications effective group.
 * @param uCtlrNum              I: If the tracer is a controller tracer, this
 *                                 parameter determines the controller to trace.
 * @param puTracerID            O: The id of the new tracer if the result is
 *                                 CAPI_OK.
 *
 * @return CAPI_OK              The new tracer was registered successfully.
 * @return Else                 An error occurred, the tracer id is not valid.
 */
extern unsigned kcapitrace_add_tracer
   (unsigned                  uApplID,
    unsigned long             ulTraceMask,
    unsigned long             ulDataBlockLength,
    int                       fIsApplTracer,
    int                       pafMayTraceAppl [CAPI_MAX_APPLICATIONS],
    unsigned                  uCtlrNum,
    unsigned                 *puTracerID);

/**
 * Remove a tracer.
 *
 + @param uTracerID             I: The id of the tracer to remove.
 *
 * @return CAPI_OK              The tracer was removed successfully.
 * @return Else                 An error occurred, no tracer was removed.
 */
extern unsigned kcapitrace_remove_tracer
   (unsigned uTracerID);

/**
 * Update the application tracer configurations for a new CAPI application.
 *
 * This function must be called when a new CAPI application is registered and
 * there is at least one tracer active (e_nNumTraces greater than 0). During
 * this call all application tracers' configuration is updated with respect to
 * the right to receive trace messages according to the new application. For
 * each tracer the user id and group ids are checked against the effective
 * user and group id of the new application's process.
 *
 * @attention The caller must ensure that no call to register or release another
 *            application or tracer can interrupt the update process. This can
 *            only be accomplished by letting the CAPI application array access
 *            be blocked during this call. The implementation of this function
 *            will never call back into the application handling module, so this
 *            locking will not leed to a deadlock.
 *
 * @param uNewApplID            I: The application id of the new application.
 * @param effUserID             I: The effective user id of the new applications
 *                                 process owner.
 * @param effGroupID            I: The effective group id of the new
 *                                 applications process owner.
 *
 * @return Nothing.
 */
extern void kcapitrace_update_tracer_for_registered_appl
   (unsigned uNewApplID,
    uid_t    effUserID,
    gid_t    effGroupID);

/**
 * Update the application tracer configurations for a released CAPI application.
 *
 * This function must be called when a CAPI application is released and there is
 * at least one tracer active (e_nNumTraces greater than 0). During this call
 * all application tracer's configuration is updated to not receive any trace
 * messages according to the released application id.
 *
 * @attention The caller must ensure that no call to register or release another
 *            application or tracer can interrupt the update process. This can
 *            only be accomplished by letting the CAPI application array access
 *            be blocked during this call. The implementation of this function
 *            will never call back into the application handling module, so this
 *            locking will not leed to a deadlock.
 *
 * @param uOldApplID            I: The application id of the removed
 *                                 application.
 *
 * @return Nothing.
 */
extern void kcapitrace_update_tracer_for_released_appl
   (unsigned uOldApplID);



/* --- tracing application API function calls --- */

/* Note: The effective user and group ids of the current process is used to
 *       check if the tracer is authorized to get informed about the respective
 *       kernel CAPI call.
 */

/**
 * Trace call to kcapi_register.
 *
 * @note This function must be called after the new application is registered by
 *       calling kcapitrace_update_tracer_for_registered_appl(). Otherwise the
 *       register call would not create any trace message.
 */
extern void kcapitrace_appl_register
   (unsigned  uMaxLogicalConnections,
    unsigned  uMaxBDataBlocks,
    unsigned  uMaxBDataLen,
    unsigned *puApplID,
    unsigned  uResult);

/**
 * Trace call to kcapi_release.
 */
extern void kcapitrace_appl_release
   (unsigned uApplID,
    unsigned uResult);

/**
 * Trace call to kcapi_put_message.
 */
extern void kcapitrace_appl_put_message
   (unsigned           uApplID,
    const struct mbuf *pmbOrgMsg,
    unsigned           uResult,
    int                fIsDataB3Msg);

/**
 * Trace call to kcapi_get_message.
 */
extern void kcapitrace_appl_get_message
   (unsigned      uApplID,
    struct mbuf **ppmbOrgMsg,
    unsigned      uResult);

/**
 * Trace call to kcapi_set_signal.
 */
#if defined(__alpha__) || defined (__ia64__) || defined (__amd64__) || defined (__sparc64__)
extern void kcapitrace_appl_set_signal
   (unsigned  uApplID,
    void (*pfnCallback) (unsigned uApplID, u_int64_t qwParam),
    u_int64_t dwParam,
    unsigned  uResult);
#else /* __alpha__ || __ia64__ || __amd64__ || __sparc64__ */
extern void kcapitrace_appl_set_signal
   (unsigned  uApplID,
    void (*pfnCallback) (unsigned uApplID, u_int32_t dwParam),
    u_int32_t dwParam,
    unsigned  uResult);
#endif /* __alpha__ || __ia64__ || __amd64__ || __sparc64__ */

/**
 * Trace call to kcapi_get_manufacturer.
 */
extern void kcapitrace_appl_get_manufacturer
   (unsigned       uCtlrNum,
    unsigned char *pszBuffer,
    unsigned       uResult);

/**
 * Trace call to kcapi_get_version.
 */
extern void kcapitrace_appl_get_version
   (unsigned uCtlrNum,
    unsigned uCAPIMajor,
    unsigned uCAPIMinor,
    unsigned uManufacturerMajor,
    unsigned uManufacturerMinor,
    unsigned uBSDMajor,
    unsigned uBSDMinor,
    unsigned uResult);

/**
 * Trace call to kcapi_get_serial_number.
 */
extern void kcapitrace_appl_get_serial_number
   (unsigned       uCtlrNum,
    unsigned char *pszBuffer,
    unsigned       uResult);

/**
 * Trace call to kcapi_get_profile.
 */
extern void kcapitrace_appl_get_profile
   (unsigned             uCtlrNum,
    CAPIProfileBuffer_t *pProfile,
    unsigned             uResult);

/**
 * Trace call to kcapi_get_ctlr_driver_info.
 */
extern void kcapitrace_appl_get_ctlr_driver_info
   (unsigned              uCtlrNum,
    CAPICtlrDriverInfo_t *pDrvInfo,
    unsigned              uResult);

/**
 * Trace call to kcapi_reset_ctlr.
 */
extern void kcapitrace_appl_reset_ctlr
   (unsigned             uCtlrNum,
    size_t               nNumDataBlocks,
    CAPICtlrDataBlock_t *paDataBlocks,
    unsigned             uResult);



/* --- tracing controller API function calls --- */

/**
 * Trace call to CAPI_REGISTER in CAPI driver.
 */
extern void kcapitrace_drvr_register
   (unsigned uUniqueCtlrNum,
    unsigned uDrvCtlrNum,
    unsigned uMaxLogicalConnections,
    unsigned uMaxBDataBlocks,
    unsigned uMaxBDataLen,
    unsigned uUniqueApplID,
    unsigned uCtlrApplID,
    unsigned uResult);

/**
 * Trace call to CAPI_RELEASE in CAPI driver.
 */
extern void kcapitrace_drvr_release
   (unsigned uUniqueCtlrNum,
    unsigned uDrvCtlrNum,
    unsigned uUniqueApplID,
    unsigned uCtlrApplID,
    unsigned uResult);

/**
 * Trace call to CAPI_PUT_MESSAGE in CAPI driver.
 */
extern void kcapitrace_drvr_put_message
   (unsigned           uUniqueCtlrNum,
    unsigned           uDrvCtlrNum,
    unsigned           uUniqueApplID,
    unsigned           uCtlrApplID,
    const struct mbuf *pmbOrgMsg,
    unsigned           uResult);

/**
 * Trace call to kcapictlr_receive_capi_message as the driver interface variant
 * of CAPI_GET_MESSAGE.
 */
extern void kcapitrace_drvr_get_message
   (unsigned           uUniqueCtlrNum, 
    unsigned           uDrvCtlrNum,
    unsigned           uUniqueApplID,
    unsigned           uCtlrApplID,
    const struct mbuf *pmbOrgMsg);

/**
 * Trace call to CAPI_RESET_CTLR in CAPI driver.
 */
extern void kcapitrace_drvr_reset_ctlr
   (unsigned             uUniqueCtlrNum,
    size_t               nNumDataBlocks,
    CAPICtlrDataBlock_t *paDataBlocks,
    unsigned             uResult);




/* --- tracing CAPI messages --- */

/**
 * Trace a CAPI message on the application interface level.
 *
 * Note: The effective user and group ids of the current process is used to
 *       check if the tracer is authorized to get this message.
 *
 * @param uApplID               I: The application id to receive or send the
 *                                 CAPI message.
 * @param pmbOrgMsg             I: The CAPI message to trace.
 * @param fIsDataB3Msg          I: 0 - The message is no Data-B3 message.
 *                                 1 - The message is a Data-B3 message.
 *
 * @return Nothing.
 */
extern void kcapitrace_trace_appl_msg
   (unsigned           uApplID,
    const struct mbuf *pmbOrgMsg,
    int                fIsDataB3Msg);

/**
 * Trace a CAPI message on the driver interface level.
 *
 + @param uUniqueCtlrNum        I: Unique controller number the message belongs
 *                                 to.
 * @param pmbOrgMsg             I: The CAPI message to trace.
 *
 * @return Nothing.
 */
extern void kcapitrace_trace_drvr_msg
   (unsigned           uUniqueCtlrNum,
    const struct mbuf *pmbOrgMsg);





/* === definition of inline functions ==================================== */





#endif /* __KCAPI_TRACE_H */
